/*
  name: tools/pkg/main.cpp

  This file is part of ARCS - Augmented Reality Component System
  (version 2-current), written by Jean-Yves Didier 
  for IBISC Laboratory (http://www.ibisc.univ-evry.fr)

  Copyright (C) 2013  Universit d'Evry-Val d'Essonne

  This program is free software: you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program.  If not, see <http://www.gnu.org/licenses/>.


  Please send bugreports  with examples or suggestions to
  jean-yves.didier__at__ibisc.univ-evry.fr
*/


#include <iostream>
#include <cstdlib>
#include <QCoreApplication>
#include <QStringList>
#include "arcspackagemanager.h"

/*! \page arcspkg arcspkg manpage
 *<H2>NAME</H2>
 *<B>arcspkg</B>
 *ARCS package manager
 *<H2>SYNOPSIS</H2>
 *<B>arcspkg </B>
 *-b|-h|-i packageName|-I [-s] [-Dsetting=value]...
 *<H2>DESCRIPTION</H2>
 *<B>arcspkg</B>
 *is a light package manager for ARCS. Packages are component libraries.
 *It can install packages from a remote source or build a special file repository
 *for packages.
 *<H2>OPTIONS</H2>
 *<DL COMPACT>
 *<DT>
 *<B>-h</B>
 *<DD>: prints a short help summary.
 *<DT>
 *<B>-b</B>
 *<DD>: builds a repository.
 *<DT>
 *<B>-i</B>
 *<I>package</I>
 *<DD>: installs the designated package
 *<DT>
 *<B>-I</B>
 *<DD>: installs all packages from remote repository.
 *<DT>
 *<B>-s</B>
 *<DD>: store settings
 *<DT>
 *<B>-Dsetting=value</B>
 *<DD>: defines the
 *<I>value </I>
 *<DD>of a particular
 *<I>setting.</I>
 *<BR>
 *Note that is the option
 *<B>-s</B>
 *is not used, the value will be set only for this run of
 *<B>arcspkg.</B>
 *</DL>
 *<H2>SETTINGS</H2>
 *<DL COMPACT>
 *<DT>
 *<B>repositoryFile</B>
 *<DD>: name of the file needed to build a repository (by default
 *<I>REPOSITORY</I>
 *)
 *<DT>
 *<B>repositoryPath</B>
 *<DD>: path where packages are stored when building a repository (see
 *<B>-b</B>
 *<DD>option)
 *<DT>
 *<B>repositoryUrl</B>
 *<DD>: remote URL where packages are stored.
 *<DT>
 *<B>compressCommand</B>
 *<DD>: command line used to compress files of the packages. Note that
 *<B>%a</B>
 *will be the name of the archive (package) and
 *<B>%p </B>
 *will be the name of the file/path to store in the archive (package).
 *By default,
 *<B>zip</B>
 *is used as a compressor.
 *<DT>
 *<B>unCompressCommand</B>
 *<DD>: command line used to uncompress files of the packages.
 *<DD>This setting also uses
 *<B>%a </B>
 *for the archive name. By default
 *<B>unzip </B>
 *is used as a decompressor.
 *<DT>
 *<B>archiveExtension</B>
 *<DD>: extension of the packages (by default
 *<I>.zip</I>
 *<DD>is used).
 *<DT>
 *<B>sourcePath</B>
 *<DD>: path where library components are stored.
 *<DD>If an environment variable named
 *<B>ARCSBUILDPATH</B>
 *exists, then this setting will take its value.
 *<BR>
 *</DL>
 *<H2>EXAMPLES</H2>
 *<DL COMPACT>
 *<DT>
 *<B>arcspkg -I</B>
 *<DD>
 *<DD>installs all available packages.
 *<DT>
 *<B>arcspkg -i toto </B>
 *<DD>
 *<DD>installs the package (component library) named toto.
 *<DT>
 *<B>arcspkg -i toto -DrepositoryURL=http://remote.org/repository/path</B>
 *<DD>
 *<DD>installs the package toto from a specific remote repository.
 *<DT>
 *<B>arcspkg -i toto -s -DrepositoryURL=http://remote.org/path/</B>
 *<DD>
 *<DD>installs the package toto from http://remote.org/path and store the
 *setting for a next use.
 *</DL>
 *<H2>NOTES</H2>
 *<P>If some settings are empty,
 *<B>arcspkg</B>
 *will automatically drop a prompt asking for the corresponding value.
 *</P>
 *<P>
 *<B>arcspkg</B>
 *supports several protocols, including http, https and ftp for retrieving remote packages.
 *</P>
 *<H2>AUTHOR</H2>
 *Jean-Yves Didier
 */



typedef enum {
    NONE,
    BUILD,
    INSTALL,
    INSTALL_ALL
} Mode;

Mode currentMode = NONE;
bool storeSettings=false;
QString libName;

void usage(ARCSPackageManager &apm, char* prgname)
{
    std::cout << " Usage: " << prgname << " -b|-i packageName|-I|-h [-s] [-Dsetting=value]" << std::endl;
    std::cout << " -h: prints this help message and current settings" << std::endl;
    std::cout << " -b: build a repository" << std::endl;
    std::cout << " -i: install packageName" << std::endl;
    std::cout << " -I: install all packages from repository" << std::endl;
    std::cout << " -s: store settings" << std::endl;
    std::cout << " -D: defines a setting with a value" << std::endl;
    apm.checkSettings();
    apm.displaySettings();
}

QString readNext(int& idx,int argc,char* argv[])
{
    idx++;
    if (idx < argc)
        return argv[idx];
    return QString::null;
}


void requestParameter(ARCSPackageManager &apm,QString paramName, QString message)
{
    if (!apm.hasSetting(paramName))
    {
        std::cout << qPrintable(message); //"Please give a path for repository: ";
        std::string rp;
        std::cin >> rp;
        apm.setSetting(paramName,rp.data());
    }
}

void parseOptions(ARCSPackageManager &apm, int argc, char* argv[])
{
    int i=1;
    QString next;

    for(i=1; i < argc; i++)
    {
         QString current(argv[i]);

         if (current == "-b")
         {
             currentMode = BUILD;
             requestParameter(apm,"repositoryPath","Please give a path for repository: ");
         }
         else {
             if (current == "-i")
             {
                 if ((next = readNext(i,argc,argv)).isEmpty())
                 {
                     std::cout << "packageName not given" << std::endl << std::endl;
                     usage(apm,argv[0]);
                     return ;
                 }
                 currentMode = INSTALL;
                 libName = next;
                 requestParameter(apm,"repositoryUrl","Please give a url for repository: ");
             }
             else
             {
                 if (current == "-h")
                 {
                     usage(apm,argv[0]);
                     return;
                 }
                 else
                 {
                     if (current == "-I")
                     {
                         currentMode = INSTALL_ALL;
                         requestParameter(apm,"repositoryUrl","Please give a url for repository: ");
                         requestParameter(apm,"repositoryFile","Please give a repository file name: ");
                     }
                     else
                     {
                         if (current == "-s")
                         {
                             storeSettings=true;
                         }
                         else
                         {
                             if (current.startsWith("-D"))
                             {
                                 QString set=current.remove(0,2);
                                 QStringList sp = set.split('=');
                                 apm.setSetting(sp[0],sp[1]);
                             }
                         }
                     }
                 }
             }
         }
    }
}


int main(int argc, char* argv[])
{
    QCoreApplication app(argc,argv);
    QCoreApplication::setApplicationName("ARCS");
    QCoreApplication::setOrganizationName("UEVE - Ibisc - IRA2 Team");
    QCoreApplication::setOrganizationDomain("ibisc.fr");

    ARCSPackageManager apm;
    if (argc < 2)
    {
        usage(apm,argv[0]);
        return 1;
    }

    if (!apm.checkSettings())
        return 2;

    parseOptions(apm,argc,argv);

    if (currentMode == BUILD)
        apm.buildRepository();
    if (currentMode == INSTALL)
        apm.installLibrary(libName);
    if (currentMode == INSTALL_ALL)
        apm.installAll();

    if (storeSettings)
        apm.storeSettings();

   return 0;
}
