/*
  name: tools/editor/maineditor.cpp

  This file is part of ARCS - Augmented Reality Component System
  (version 2-current), written by Jean-Yves Didier 
  for IBISC Laboratory (http://www.ibisc.univ-evry.fr)

  Copyright (C) 2013  Universit d'Evry-Val d'Essonne

  This program is free software: you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program.  If not, see <http://www.gnu.org/licenses/>.


  Please send bugreports  with examples or suggestions to
  jean-yves.didier__at__ibisc.univ-evry.fr
*/


#include "maineditor.h"
#include "applicationview.h"
#include "logview.h"
#include <QMenuBar>
#include <QApplication>
#include <QPushButton>
#include <QTabWidget>
#include <QTextEdit>
#include <QSettings>
#include <QFileDialog>
#include "arcs/arcsxmlhandler.h"
#include <QDebug>
#include "arcs/arcslog.h"

MainEditor::MainEditor(QWidget *parent) :
    QMainWindow(parent)
{

    defaultApplication = 0;
    recentMenu = new QMenu("Recent");

    QSettings settings;
    recentList = settings.value("recent files").toStringList();
    buildRecentMenu();
    connect(recentMenu,SIGNAL(triggered(QAction*)),this,SLOT(handleRecentAction(QAction*)));


    QMenu* fileMenu = new QMenu("File",menuBar());
    fileMenu->addAction("New...",this,SLOT(newApplication()),QKeySequence(Qt::CTRL + Qt::Key_N));
    fileMenu->addAction("Open",this,SLOT(openApplication()),QKeySequence(Qt::CTRL + Qt::Key_O));
    saveFileAction=fileMenu->addAction("Save",this, SLOT(saveApplication()),QKeySequence(Qt::CTRL + Qt::Key_S));
    saveFileAction->setEnabled(false);
    fileMenu->addAction("Save as...",this,SLOT(saveAsApplication()));
    fileMenu->addMenu(recentMenu);
    fileMenu->addSeparator();
    fileMenu->addAction("Exit",QApplication::instance(),SLOT(quit()),QKeySequence(Qt::CTRL + Qt::Key_Q));

    //menuBar()->addAction("New...");
    editMenu = new QMenu("Edition",menuBar());


    menuBar()->addMenu(fileMenu);
    menuBar()->addMenu(editMenu);

    tbElements = new QTabWidget(this);

    setCentralWidget(tbElements);

    welcomeWidget = new QTextEdit(this);

    welcomeWidget->setHtml("<img src=\":arcs_logo.png\"/><h1>Welcome to the ARCS Editor !</h1><p>You can start a new project by clicking <tt>File -> New...</tt>");
    welcomeWidget->setReadOnly(true);


    tbElements->addTab(welcomeWidget,"Welcome");
    tbElements->setTabsClosable(true);
    connect(tbElements,SIGNAL(tabCloseRequested(int)),this,SLOT(removeTab(int)));
    connect(tbElements,SIGNAL(currentChanged(int)),this,SLOT(actualizeFromTab(int)));
    setWindowIcon(QIcon(":arcs_logo.png"));

    logView = new LogView(this);
    addDockWidget(Qt::BottomDockWidgetArea,logView);

}

MainEditor::~MainEditor()
{
    QSettings settings;
    settings.setValue("recent files",QStringList(recentList.mid(0,5)));
}



void MainEditor::newApplication()
{
    ApplicationView* app = new ApplicationView(0,this);
    applications.append(app);
    addDockWidget(Qt::LeftDockWidgetArea,app);
    connect(app,SIGNAL(defaultOwnershipRequested(ApplicationView*)),this,SLOT(setDefaultApplication(ApplicationView*)));
    connect(app,SIGNAL(requestAppIdDestroy(QVariant)),this,SLOT(destroyAppId(QVariant)));
    connect(app,SIGNAL(requestWidgetDestroy(QVariant,QString)),this,SLOT(destroyWidget(QVariant,QString)));
    setDefaultApplication(app);
}


void MainEditor::saveApplication()
{
    if (!defaultApplication)
        return;

    // put here something in order to quick save all of this.

}

void MainEditor::saveAsApplication()
{
    QString fileName = QFileDialog::getSaveFileName(this,"Save your ARCS application");

    if (fileName.isEmpty())
        return ;

    if (recentList.contains(fileName))
        recentList.swap(0,recentList.indexOf(fileName));
    else
        recentList.push_front(fileName);
    buildRecentMenu();


    // look for dock widgets.
    if (defaultApplication)
    {
        ARCSLog::getInstance()->startCapture();
        // another idea is to delegate the fact we save the application to the application viewer.
        ARCSApplicationComponent* aac = defaultApplication->getApplication();
        //aac->saveFile(fileName);
        ARCSXMLHandler handler;
        //handler.pushFilePath(fileName);
        handler.storeApplication(aac);
        handler.saveFile(fileName);

        logView->addLog(ARCSLog::getInstance()->endCapture());
    }
}

void MainEditor::openApplication(QString s)
{
    QString fileName ;
    if (s.isEmpty())
    {
        fileName = QFileDialog::getOpenFileName(this,"Please select your ARCS application");
        if (fileName.isEmpty())
            return ;

        if (recentList.contains(fileName))
            recentList.swap(0,recentList.indexOf(fileName));
        else
            recentList.push_front(fileName);
        buildRecentMenu();

    }
    else
        fileName = s;

    ARCSLog::getInstance()->startCapture();


    ARCSApplicationComponent* appComp = new ARCSApplicationComponent();
    if (appComp->loadFile(fileName))
    {
        ApplicationView* app = new ApplicationView(appComp,this);
        app->setAssociatedFile(fileName);
        applications.append(app);
        app->setWindowTitle("Application: " + QFileInfo(fileName).fileName());
        addDockWidget(Qt::LeftDockWidgetArea,app);
        connect(app,SIGNAL(defaultOwnershipRequested(ApplicationView*)),this,SLOT(setDefaultApplication(ApplicationView*)));
        connect(app,SIGNAL(requestWidgetDestroy(QVariant,QString)),this,SLOT(destroyWidget(QVariant,QString)));
        connect(app,SIGNAL(requestAppIdDestroy(QVariant)),this,SLOT(destroyAppId(QVariant)));
        setDefaultApplication(app);
    }


    logView->addLog(ARCSLog::getInstance()->endCapture());
}


void MainEditor::addNewTabWidget(QWidget *wdg)
{
    //! \todo check to avoid double
    for (int i=0; i < tbElements->count(); i++)
    {
        if (tbElements->tabText(i) == wdg->windowTitle() && wdg->property("appId") == tbElements->widget(i)->property("appId"))
        {
            tbElements->setCurrentIndex(i);
            delete wdg;
            return;
        }
    }

    tbElements->addTab(wdg,wdg->windowTitle());
    tbElements->setCurrentIndex(tbElements->count()-1);
    connect(wdg,SIGNAL(sendLog(QString)),logView,SLOT(addLog(QString)));
    connect(wdg,SIGNAL(addWidget(QWidget*)),this,SLOT(addNewTabWidget(QWidget*)));

}

void MainEditor::actualizeFromTab(int cur)
{
    if (cur < 0)
        return ;

    editMenu->clear();
    editMenu->addActions(tbElements->currentWidget()->actions());
}

void MainEditor::handleRecentAction(QAction * act)
{
    openApplication(act->text());
}

void MainEditor::buildRecentMenu()
{
    for (int i=0; i < recentList.count(); i++)
        recentMenu->addAction(recentList[i]);
}

void MainEditor::setDefaultApplication(ApplicationView *app)
{
    if (defaultApplication)
        defaultApplication->setStyleSheet("QDockWidget { font-weight: normal; }");
    defaultApplication = app;
    saveFileAction->setEnabled(!app->getAssociatedFile().isEmpty());
    app->setStyleSheet("QDockWidget { font-weight: bold; } ");
}

void MainEditor::destroyAppId(QVariant appId)
{
    int id = 0;
    for (int i = 0; i < applications.count(); i++)
    {
        if (applications[i]->property("appId") != appId)
            setDefaultApplication(applications[i]);
        else
            id = i;
    }

    applications.removeAt(id);
    if (applications.count() == 0)
    {
        saveFileAction->setEnabled(false);
        defaultApplication = 0;
    }

    for (int i=0; i < tbElements->count(); i++)
    {
        if( tbElements->widget(i)->property("appId") == appId)
            tbElements->removeTab(i);
    }
}

void MainEditor::destroyWidget(QVariant appId, QString name)
{
    for (int i=0; i < tbElements->count(); i++)
    {
        if( tbElements->widget(i)->property("appId") == appId && tbElements->widget(i)->windowTitle() == name)
            tbElements->removeTab(i);
    }
}
