/*
  name: tools/editor/graphicsinvocationitem.cpp

  This file is part of ARCS - Augmented Reality Component System
  (version 2-current), written by Jean-Yves Didier 
  for IBISC Laboratory (http://www.ibisc.univ-evry.fr)

  Copyright (C) 2013  Universit d'Evry-Val d'Essonne

  This program is free software: you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program.  If not, see <http://www.gnu.org/licenses/>.


  Please send bugreports  with examples or suggestions to
  jean-yves.didier__at__ibisc.univ-evry.fr
*/


#include "graphicsinvocationitem.h"
#include <QBrush>
#include <QStyleOptionGraphicsItem>
#include <QGraphicsRectItem>
#include <QGraphicsSimpleTextItem>
#include <iostream>
#include "sheetview.h"

GraphicsInvocationItem::GraphicsInvocationItem(QGraphicsItem *parent,GraphicsInvocationItem::InvocationType itype, ARCSInit &i) :
    QGraphicsPathItem(parent), init(i)
{
#if QT_VERSION >= 0x050000
    setAcceptHoverEvents(true);
#else
    setAcceptsHoverEvents(true);
#endif
    setFlag(ItemIsSelectable,true);
    invocationType = itype;
    defaultPen = pen();
    drawPath();
    markedForDeletion = false;
    rectInvocation = new QGraphicsRectItem(this);
    textInvocation = new QGraphicsSimpleTextItem(rectInvocation);
    textNumber = new QGraphicsSimpleTextItem(this);
    rectInvocation->setVisible(false);
    QString invocationTypeName;

    switch(invocationType)
    {
    case Pre:
        invocationTypeName = "Pre-connection" ;
        break;
    case Post:
        invocationTypeName = "Post-connection" ;
        break;
    case Cleanup:
        invocationTypeName = "Clean-up invocation";
        break;
    }


    textInvocation->setText(invocationTypeName +
                            "\nType:\"" + init.getValueType() +
                            "\";\nValue:\"" + init.getValueString() + "\"");
    rectInvocation->setBrush(QBrush(Qt::lightGray));
    rectInvocation->setRect(textInvocation->boundingRect().adjusted(-2,-2,2,2));
    QSizeF s = rectInvocation->rect().size();
    rectInvocation->setPos(-s.width()-15,10);
}


GraphicsInvocationItem::~GraphicsInvocationItem()
{
    setSelected(false);
    if (markedForDeletion)
    {
        SheetView* view = dynamic_cast<SheetView*>(scene()->views()[0]);
        if (!view)
            return ;

        std::cout << "We shall remove this initialisation from sheet" << std::endl;

        switch(invocationType)
        {
        case Pre:
            view->getSheet().removePreconnect(init);
            break;
        case Post:
            view->getSheet().removePostconnect(init);
            break;
        case Cleanup:
            view->getSheet().removeCleanup(init);
            break;
        }
    }
}



void GraphicsInvocationItem::drawPath()
{
    QPainterPath pp;

    switch(invocationType)
    {
    case Pre: // then draw an event
    {
        QBrush br(Qt::white);
        setBrush(br);
        pp.moveTo(-5,0);
        pp.lineTo(-10,0);
        pp.lineTo(-12.5,-5);
        pp.lineTo(-20,-5);
        pp.lineTo(-17.5,0);
        pp.lineTo(-20,5);
        pp.lineTo(-12.5,5);
        pp.lineTo(-10,0);
        setPath(pp);
        break;
    }
    case Post: // then draw a filled circle
    {
        QBrush br(Qt::black);
        setBrush(br);
        pp.moveTo(-5,0);
        pp.lineTo(-10,0);
        //pp.moveTo(-30,0);
        pp.addEllipse(QPointF(-15,0),5,5);
        setPath(pp);
        break;
    }
    case Cleanup: // then draw a cross circled
    {
        QBrush br(Qt::white);
        setBrush(br);
        pp.moveTo(-5,0);
        pp.lineTo(-10,0);
        pp.addEllipse(QPointF(-15,0),5,5);
        pp.moveTo(-11.47,3.03);
        pp.lineTo(-18.53,-3.03);
        pp.moveTo(-18.53,3.03);
        pp.lineTo(-11.47,-3.03);
        setPath(pp);
        break;
    }
    }
}


void GraphicsInvocationItem::paint(QPainter *painter, const QStyleOptionGraphicsItem *option, QWidget *widget)
{
    QStyleOptionGraphicsItem myOptions(*option);
    if (myOptions.state & (QStyle::State_Selected))
    {
        QPen myPen;
        myPen.setWidth(2);
        myPen.setColor(Qt::darkRed);
        setPen(myPen);
        myOptions.state = myOptions.state ^ QStyle::State_Selected  ;
    }
    else
    {
        setPen(defaultPen);
    }
    QGraphicsPathItem::paint(painter,& myOptions,  widget );

}

void GraphicsInvocationItem::hoverEnterEvent(QGraphicsSceneHoverEvent */*event*/)
{
    SheetView* view = dynamic_cast<SheetView*>(scene()->views()[0]);
    if (!view)
        return ;

    int idx = 0;
    switch(invocationType)
    {
    case Pre:
        idx = view->getSheet().getPreconnectIndex(init);
        break;
    case Post:
        idx = view->getSheet().getPostconnectIndex(init);
        break;
    case Cleanup:
        idx = view->getSheet().getCleanupIndex(init);
        break;
    }

    textNumber->setText(QString::number(idx));
    QFont font = textNumber->font();
    font.setBold(true);
    textNumber->setFont(font);
    textNumber->setBrush(QBrush(Qt::darkMagenta));
    QSizeF s = textNumber->boundingRect().size();
    textNumber->setPos(-s.width()-25,-s.height()/2);

    textNumber->setVisible(true);
    rectInvocation->setVisible(true);
}

void GraphicsInvocationItem::hoverLeaveEvent(QGraphicsSceneHoverEvent */*event*/)
{
    textNumber->setVisible(false);
    rectInvocation->setVisible(false);
}

