/*
  name: tools/arcs1to2/main.cpp

  This file is part of ARCS - Augmented Reality Component System
  (version 2-current), written by Jean-Yves Didier 
  for IBISC Laboratory (http://www.ibisc.univ-evry.fr)

  Copyright (C) 2013  Universit d'Evry-Val d'Essonne

  This program is free software: you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program.  If not, see <http://www.gnu.org/licenses/>.


  Please send bugreports  with examples or suggestions to
  jean-yves.didier__at__ibisc.univ-evry.fr
*/



/*! \page arcs1to2 arcs1to2 manpage
 *<H2>NAME</H2>
 *<B>arcs1to2 </B> tool to migrate ARCS applications and component libraries from version 1
 *of the framework to version 2.
 *<H2>SYNOPSIS</H2>
 *<B>arcs1to2</B> OPTION FILENAME
 *<H2>DESCRIPTION</H2>
 *<B>arcs1to2</B> parses different file formats associated to the first version of the ARCS
 *framework and translates them to an equivalent in the second version.
 *Translated files may be: application files, composite component files,
 *library description files and project files.
 *<H2>OPTIONS</H2>
 *<DL COMPACT>
 *<DT>
 *<B>-a</B>
 *<DD>file is an XML formatted application description
 *<DT>
 *<B>-c </B>
 *<DD>file is an XML formatted composite component description
 *<DT>
 *<B>-l</B>
 *<DD>file is a cpp file (usually named
 *<I>lib.cpp</I>) to translate into an XML component library description
 *<DT>
 *<B>-p</B>
 *<DD>file is a project file. This is the most complete mode since it also handles
 *and convert source files of the project (old versions of library descriptions
 *are also processed).
 *</DL>
 *<H2>AUTHOR</H2>
 *Jean-Yves Didier
 */




#include <QCoreApplication>
#include <iostream>
#include <QXmlQuery>
#include <QXmlFormatter>
#include <QTextStream>
#include <QTemporaryFile>
#include <QDomDocument>
#include <QProcess>
#include <QStringList>
#include <QFile>
#include <QDir>
#include <QRegExp>


QString qmake_append =
"\n\nALXFILE = @\nOTHER_FILES += @\narcslibrary.output = alm_${QMAKE_FILE_BASE}.cpp\n" \
"arcslibrary.input = ALXFILE\narcslibrary.commands = arcslibmaker ${QMAKE_FILE_NAME}\n" \
"arcslibrary.variable_out = SOURCES\n" \
"QMAKE_EXTRA_COMPILERS += arcslibrary\n";

QString qmake_preamble =
     "INCLUDEPATH += $$(ARCSDIR)/include\n" \
     "LIBS += -L$$(ARCSDIR)/lib -larcs\n "\
     "CONFIG += dll\n";

QStringList capList(QString pattern,QString input)
{
     QStringList res;
     QRegExp regexp(pattern);
     regexp.setMinimal(true);

     int pos= 0;
     while ((pos = regexp.indexIn(input, pos)) != -1) {
          res << regexp.cap(1) ;
           pos += regexp.matchedLength();
     }
     return res;
}


bool isLibraryFile(QString fileName)
{
     QFile file(fileName);
     if (!file.open(QIODevice::ReadOnly))
          return false;
     bool res = QTextStream(&file).readAll().contains(QRegExp("\\sMETALIB_BEGIN\\s"));
     file.close();
     return res;
}

void processHeader(QString fileName)
{
     //std::cout << "Processing " << qPrintable(fileName) << std::endl;
     QFile input(fileName);
     if (!input.open(QIODevice::ReadOnly))
     {
          std::cerr << "Failed to read " << qPrintable(fileName) << std::endl;
          return ;
     }
     QString contents = QTextStream(&input).readAll();
     input.close();

     QRegExp constructFilter("[(]\\s*QObject\\s*\\*\\s*(\\w*)\\s*=\\s*0\\s*,\\s*const\\s*char\\s*\\*\\s*(\\w*)\\s*=\\s*0\\s*[)]\\s*:\\s*(\\w*)\\s*[(]\\s*\\1\\s*,\\s*\\2\\s*[)]");
     constructFilter.setMinimal(true);
     int pos=0;

     bool res=false;
     while ((pos = constructFilter.indexIn(contents,pos)) != -1 ) {
          contents.replace(pos,constructFilter.matchedLength(), "(QObject* "+constructFilter.cap(1)
                           +"=0) : " + constructFilter.cap(3) + "(" + constructFilter.cap(1) + ")" );
          res =true;
     }

     constructFilter.setPattern("[(]\\s*QObject\\s*\\*\\s*(\\w*)\\s*=\\s*0\\s*,\\s*const\\s*char\\s*\\*\\s*(\\w*)\\s*=\\s*0\\s*[)]");
     pos=0;

     while ((pos = constructFilter.indexIn(contents,pos)) != -1 ) {
          contents.replace(pos,constructFilter.matchedLength(), "(QObject* "+constructFilter.cap(1)+"=0)");
          res =true;
     }

     if (res)
          std::cout << "Modifying " << qPrintable(fileName) << std::endl;
     else
          return;


     QFile output(fileName);
     if (! output.open(QIODevice::WriteOnly))
     {
          std::cerr << "Failed to write " << qPrintable(fileName) << std::endl;
          return ;
     }
     QTextStream(&output) << contents ;
     output.close();
}


void processCppSource(QString fileName)
{
     //std::cout << "Processing " << qPrintable(fileName) << std::endl;
     QFile input(fileName);
     if (!input.open(QIODevice::ReadOnly))
     {
          std::cerr << "Failed to read " << qPrintable(fileName) << std::endl;
          return ;
     }
     QString contents = QTextStream(&input).readAll();
     input.close();

     QRegExp constructFilter("[(]\\s*QObject\\s*\\*\\s*(\\w*)\\s*,\\s*const\\s*char\\s*\\*\\s*(\\w*)\\s*[)]\\s*:\\s*(\\w*)\\s*[(]\\s*\\1\\s*,\\s*\\2\\s*[)]");
     constructFilter.setMinimal(true);
     int pos=0;

     bool res=false;
     while ((pos = constructFilter.indexIn(contents,pos)) != -1 ) {
          contents.replace(pos,constructFilter.matchedLength(), "(QObject* "+constructFilter.cap(1)
                           +") : " + constructFilter.cap(3) + "(" + constructFilter.cap(1) + ")" );
          res =true;
     }
     if (res)
          std::cout << "Modifying " << qPrintable(fileName) << std::endl;
     else
          return;


     QFile output(fileName);
     if (! output.open(QIODevice::WriteOnly))
     {
          std::cerr << "Failed to write " << qPrintable(fileName) << std::endl;
          return ;
     }
     QTextStream(&output) << contents ;
     output.close();
}


void portLibraryFile(QString fileName,QString outName= QString::null)
{
     QFile input(fileName);
     if (!input.open(QIODevice::ReadOnly))
          return ;
     QString contents = QTextStream(&input).readAll();
     input.close();

     QDomDocument doc;
     QDomElement root = doc.createElement("library");
     doc.appendChild(root);
     QDomElement headers = doc.createElement("headers");
     QDomElement components = doc.createElement("components");
     root.appendChild(headers);
     root.appendChild(components);

     // parsing headers
     QRegExp regExp("#include\\s*\"(.+)\"");
     regExp.setMinimal(true);
     int pos = 0;
      while ((pos = regExp.indexIn(contents, pos)) != -1) {
           QDomElement header = doc.createElement("header");
           header.setAttribute("name", regExp.cap(1));
           headers.appendChild(header);
           pos += regExp.matchedLength();
      }

      //parsing components
      regExp.setPattern("METALIB_OBJECT\\s*[(](.+)[)]");
      pos = 0;
       while ((pos = regExp.indexIn(contents, pos)) != -1) {
            QDomElement component = doc.createElement("component");
            component.setAttribute("name", regExp.cap(1));
            components.appendChild(component);
            pos += regExp.matchedLength();
       }

       QFile* output;

       if (!outName.isEmpty())
       {
            output=new QFile(outName);
            if (! output->open(QIODevice::WriteOnly) )
            {
                 std::cerr << "Could not create ARCS2 libfile " << qPrintable(outName) << std::endl;
                 return ;
             }
       }
       else
       {
            output = new QFile();
            if(!output->open(1,QIODevice::WriteOnly))
            {
                 std::cerr << "Could not open stdout ?!?" << std::endl;
                 return ;
            }
       }

       QTextStream ts(output);
       doc.save(ts,4);
       output->close();
       delete(output);
}


void portApplicationFile(QString fileName)
{
     QXmlQuery query;
     QFileInfo fi(fileName);
     QFile file(fi.baseName()+".arcs2."+fi.suffix());
     query.bindVariable("file",QVariant(QDir::currentPath()+ QDir::separator()+fileName));
     query.setQuery(QUrl("qrc:///arcsapp1to2.xq"),QDir::currentPath());
     file.open(QIODevice::WriteOnly);
     QXmlFormatter fmt(query, &file);
     query.evaluateTo(&fmt);
     file.close();
}


void portComponentFile(QString fileName)
{
     QXmlQuery query;
     QFileInfo fi(fileName);
     QFile file(fi.baseName()+".arcs2."+fi.suffix());
     query.bindVariable("file",QVariant(QDir::currentPath()+ QDir::separator()+fileName));
     query.setQuery(QUrl("qrc:///arcscmp1to2.xq"),QDir::currentPath());
     file.open(QIODevice::WriteOnly);
     QXmlFormatter fmt(query, &file);
     query.evaluateTo(&fmt);
     file.close();
}


void portProjectFile(QString fileName)
{
     std::cout << "Porting from Qt3 to Qt4" << std::endl;
     std::cout << "============================================" << std::endl;

     // port Qt3 -> Qt4  raliser en premier

     // prparer le fichier temporaire pour faire le port
     QTemporaryFile* portFile= QTemporaryFile::createLocalFile(":/q3porting_arcs.xml");
     // Lancer le process qt3to4

     QProcess process;
     process.setProcessChannelMode(QProcess::ForwardedChannels);;
     QStringList args;
     args << "-rulesFile" <<portFile->fileName()  << "-alwaysOverwrite" << fileName ;
     process.start("qt3to4",args );

     process.waitForFinished(-1);
     delete portFile;

     std::cout << "============================================" << std::endl << std::endl;
     std::cout << "Porting from ARCS1 to ARCS2" << std::endl;
     std::cout << "============================================" << std::endl;


     // a partir d'ici le port qt3 -> qt4 est thoriquement ralis ...

     // couche spcifique ARCS2
     QFile input(fileName);
     if (!input.open(QIODevice::ReadOnly))
          return ;
     QString project = QTextStream(&input).readAll();

     input.close();


     project.insert(0,"# Project modified by ARCS1to2\n\n");
     // retrouver le nom de la cible.
     QString target;
     QRegExp regexp("\\bTARGET\\s*=\\s*(\\w+)\\b");
     regexp.setMinimal(true);
     int pos=regexp.indexIn(project);
     if (pos > -1)
          target= regexp.cap(1);
     // sinon positionner target au nom de projet !
     if (target.isEmpty())
          target=QFileInfo(fileName).baseName();
     std::cout << "Target is " << qPrintable(target) << std::endl;


     // retrouver les fichiers cpp et surtout celui qui contient la dfinition de la bibliothque.
     QStringList cppSources=capList("\\s*(\\w+[.]cpp)\\s*",project);
     for (int i=0; i < cppSources.count(); i++)
     {
          if( isLibraryFile(cppSources[i]))
          {
               std::cout << qPrintable(cppSources[i]) << " is the library file !"<< std::endl;
               project.replace(QRegExp("\\b"+cppSources[i]+"\\b"),"");
               QString alxFile = "lib"+QFileInfo(fileName).baseName()+".alx";
               portLibraryFile(cppSources[i], alxFile);
               project.append(qmake_append.replace("@",alxFile));
          }
          else
          {
               processCppSource(cppSources[i]);
          }
     }

     QStringList headers = capList("\\s*(\\w+[.]h)\\s*",project);
     for (int i=0; i< headers.count(); i++)
     {
          processHeader(headers[i]);
     }

     project.append(qmake_preamble);

     std::cout << "============================================" << std::endl << std::endl;
     std::cout << "Writing down final project file" << std::endl;

     QFile output(fileName);
     if (!output.open(QIODevice::WriteOnly))
     {
          std::cerr << "Cannot write down final project file " << std::endl;
          return ;
     }

     QTextStream(&output) << project ;
     output.close();

//     std::cout << qPrintable(project) << std::endl;
}




bool parseOptions(int argc,char* argv[])
{
     for (int i=1; i < argc; i++)
     {
          QString current(argv[i]);
          if (current == "-a")
               portApplicationFile(argv[++i]);
          else
          {
               if(current== "-c")
                    portComponentFile(argv[++i]);
               else
               {
                    if (current == "-l")
                         portLibraryFile(argv[++i]);
                    else
                         if (current == "-p")
                              portProjectFile(argv[++i]);
               }
          }

     }
     return true;
}



void usage(char* name)
{
     std::cout << "Usage: " << name << " (-a|-c|-l|-p) filename " << std::endl;
     std::cout << "\t -a stands for application files in XML format;" << std::endl;
     std::cout << "\t -c stands for composite component files in XML format;" << std::endl;
     std::cout << "\t -l stands for library files in CPP format;" << std::endl;
     std::cout << "\t -p stands for project files in pro format;" << std::endl;
}



int main(int argc,char* argv[])
{
     QCoreApplication app(argc,argv);
     std::cout << argv[0] << ", a porting tool from ARCS1 to ARCS2"  << std::endl;

     if (argc != 3)
     {
          usage(argv[0]);
          return 0;
     }

     if (!parseOptions(argc,argv))
     {
          usage(argv[0]);
          return 0;
     }


     return 0;
}
