/*
  name: lib/arcsxmlhandler.cpp

  This file is part of ARCS - Augmented Reality Component System
  (version 2-current), written by Jean-Yves Didier 
  for IBISC Laboratory (http://www.ibisc.univ-evry.fr)

  Copyright (C) 2013  Universit d'Evry-Val d'Essonne

  This program is free software: you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program.  If not, see <http://www.gnu.org/licenses/>.


  Please send bugreports  with examples or suggestions to
  jean-yves.didier__at__ibisc.univ-evry.fr
*/


#include <arcs/arcsxmlhandler.h>

#include <arcs/arcsstatemachine.h>
#include <arcs/arcsapplicationcomponent.h>
#include <arcs/arcscompositecomponent.h>
#include <arcs/arcs.h>
#include <arcs/arcslog.h>

#include <iostream>

#include <QFile>
#include <QFileInfo>
#include <QTextStream>
#include <QDir>



ARCSXMLHandler::ARCSXMLHandler()
{
     factory = ARCSFactory::getInstance();
}

ARCSXMLHandler::~ARCSXMLHandler()
{
}


bool ARCSXMLHandler::openFile(QString fileName)
{
    QFileInfo fi(fileName);
    this->fileName = fi.canonicalFilePath();

    if (!fi.exists())
    {
        ARCSLog::logCritical(logSource(), "file " + this->fileName + " does not exist.");
        return false;
    }


     QFile file(this->fileName);

     if (!file.open(QIODevice::ReadOnly))
     {
          ARCSLog::logCritical(logSource(),"cannot open file " +
                               this->fileName);
          return false;
     } 

     QString error;
     int line, col;
     bool ok = doc.setContent(&file, &error, &line, &col);

     if(!ok) 
     {
          ARCSLog::logCritical(logSource(line),
                               QString("malformed xml: (line: ")+ line +
                               ", col:" + col + ") "+ error);
          return false;
     }

     file.close();
     reset();
     return true;
}


bool ARCSXMLHandler::setContents(QString contents)
{
     QString error;
     int line, col;

     bool ok = doc.setContent(contents, &error, &line, &col);


     if(!ok) 
     {
          ARCSLog::logCritical(logSource(line), QString("malformed xml: (line: ")+ line +", col:" + col + ") "+ error);
          return false;
     }

     reset();
     return true;
}


QString ARCSXMLHandler::getContents()
{
     return doc.toString();
}


bool ARCSXMLHandler::saveFile(QString s)
{
    QFileInfo fi(s);
    this->fileName = QDir::cleanPath(fi.absoluteFilePath());
    QFile file(this->fileName);

     if (!file.open(QIODevice::WriteOnly))
     {
         std::cerr << "[SaveFile] Cannot open file " <<  qPrintable(this->fileName) << std::endl;
	  return false;
     }

     computeRelativePaths(this->fileName);

     QTextStream ts(&file);

     doc.save(ts,4);
     ts.flush();
     if (ts.status() != QTextStream::Ok)
	  return false;

     file.close();
     return true;
}


bool ARCSXMLHandler::parseContext(ARCSContext* ctx)
{
     int i;

     ctx->reset();

     QDomNodeList contexts = doc.elementsByTagName(QString("context"));

     if (contexts.isEmpty())
     {
          ARCSLog::logWarning(logSource(),"\"context\" tag undefined.");
          return true;
     }

     if (!contexts.at(0).hasChildNodes())
     {
          ARCSLog::logWarning(logSource(contexts.at(0).lineNumber()),"\"context\" tag is empty.");
          return true;
     }

     QDomNodeList contextChildren = contexts.at(0).childNodes();

     for (i=0; i < contextChildren.count() ; i++)
     {
          QDomNode e = contextChildren.at(i);
          if (e.isElement())
          {
               QDomElement elt = e.toElement();
               if (elt.tagName() == "libraries" )
               {
                   bool ok;
                   parseLibraries(elt,&ok);
                   if (!ok)
                       return false;

               }
               if (elt.tagName() == "components" )
                    parseComponents(elt,ctx);
               if (elt.tagName() == "constants" )
                    parseConstants(elt,ctx);
          }
     }
     return true;
}

//! \todo complete this parsing with the appropriate return values
bool ARCSXMLHandler::parseSheet(ARCSSheet* sheet, QDomElement father)
{
     int i,j,k;
     // father contient dj l'lment sheet
     if (father.hasAttribute("id"))
          sheet->setName(father.attribute("id"));


     QDomNodeList children = father.childNodes();

     for(i = 0; i< children.count(); i++)
     {
          QDomElement element = children.at(i).toElement();
          if (element.tagName() == "connections")
          {
               QDomNodeList eltChildren = element.childNodes();
               for (j=0; j < eltChildren.count(); j++)
               {
                    QDomElement elt = eltChildren.at(j).toElement();
                    if (elt.tagName() == "link" )
                    {
                         if (elt.hasAttribute("source") &&
                             elt.hasAttribute("destination") &&
                             elt.hasAttribute("signal") &&
                             elt.hasAttribute("slot") )
                         {
                              bool b = false;
                              if (elt.hasAttribute("queued"))
                              {
                                   QString sQueued = elt.attribute("queued").toLower();
                                   if (sQueued == "t" || sQueued == "1" || sQueued == "true")
                                        b = true;
                              }

                              ARCSConnection& connect = sheet->addConnection(elt.attribute("source"), elt.attribute("signal"),
                                                                             elt.attribute("destination"), elt.attribute("slot"),b);
                              if (!connect.isComplete())
                                   ARCSLog::logError(logSource(elt.lineNumber()),"could not create complete link." );

                              //! \todo insert here something for link coordinates
                              if (elt.hasAttribute("coords"))
                              {
                                  QList<QPointF> coords;
                                  QStringList coordStrings = elt.attribute("coords").split(' ');
                                  for (k=0; k < coordStrings.count(); k++)
                                  {
                                      coords.append(QPointF(coordStrings[k].section(',',0,0).toFloat(),
                                                            coordStrings[k].section(',',1,1).toFloat()));
                                  }
                                  connect.setCoordinates(coords);

                              }

                         }
                         else
                              ARCSLog::logWarning(logSource(elt.lineNumber()),"\"link\" tag is missing either \"source\", \"destination\", \"signal\" or \"slot\" attributes.");
                    }
               }
          }
          if (element.tagName() == "preconnections")
               parseInvocation(element, sheet, &ARCSSheet::addPreconnect);

          if (element.tagName() == "postconnections")
               parseInvocation(element, sheet, &ARCSSheet::addPostconnect);

          if (element.tagName() == "cleanups")
               parseInvocation(element, sheet, &ARCSSheet::addCleanup);

          if (element.tagName() == "properties")
              parseSheetProperties(element,sheet);

     }

     return true;
}


void ARCSXMLHandler::parseSheetProperties(QDomElement elt, ARCSSheet *sheet)
{
    int j;
    QDomNodeList eltChildren = elt.childNodes();
    for (j=0; j< eltChildren.count(); j++)
    {
        QDomElement element = eltChildren.at(j).toElement();
        if (element.tagName() == "property")
        {
            if (element.hasAttribute("id"))
                sheet->setProperty(element.attribute("id"),nodeInternalsToText(element));
            else
                ARCSLog::logError(logSource(element.lineNumber())," \"property\" tag is missing \"id\" attribute.");
        }
    }
}



//! \todo voir le probleme sur le pointeur de fonctions
void ARCSXMLHandler::parseInvocation(QDomElement element,ARCSSheet* sheet,  ARCSSheet::AddInvocationMethod method)
{
     int j;
     QDomNodeList eltChildren = element.childNodes();
     for (j=0; j< eltChildren.count(); j++)
     {
	  QDomElement elt = eltChildren.at(j).toElement();
       if (elt.tagName() == "invoke")
       {
            if (elt.hasAttribute("destination") &&
               elt.hasAttribute("slot") &&
                elt.hasAttribute("type") )
            {
                 if (!(sheet->*method)(elt.attribute("destination"), elt.attribute("slot"),
                     elt.attribute("type"), nodeInternalsToText(elt)).isComplete())
                 {
                      ARCSLog::logError(logSource(elt.lineNumber()),"could not create invocation.");
                 }
            }
            else
                 ARCSLog::logError(logSource(elt.lineNumber()),"\"invoke\" tag is missing either \"destination\", \"slot\" or \"type\" attribute." );
           }
     }
}

QStringList ARCSXMLHandler::parseLibraries(QDomElement libraries,bool* ok)
{
    QStringList result;

    QDomNodeList librariesList = libraries.childNodes();
    if (ok)    *ok = true;

    for(int i = 0 ; i < librariesList.count(); i++)
    {
        if (librariesList.at(i).isElement())
        {
            QDomElement element = librariesList.at(i).toElement();
            if( element.tagName() == "library")
            {
                if(element.hasAttribute("path"))
                {
                    QString path = element.attribute("path");
                    QString finalPath = solveRelativePath(path) ; //pushPath?pathStack.topAbsolutePathWith(path):path;

                    if (finalPath.isEmpty())
                    {
                        ARCSLog::logError(logSource(element.lineNumber()), "library "+finalPath + " not found.");
                        if (ok) *ok = *ok && false;
                    }
                    else
                    {
                        if (factory->loadLibrary(finalPath))
                            result << path ;
                        else
                        {
                            ARCSLog::logError(logSource(element.lineNumber()), "failed to load library "+finalPath);
                            if (ok) *ok = *ok && false;
                        }
                    }
                }
                else
                    ARCSLog::logWarning(logSource(element.lineNumber()),"\"library\" tag is missing \"path\" attribute.");
            }
        }
    }
    return result;
}


bool ARCSXMLHandler::parseStateMachine(ARCSStateMachine* sm)
{
     int i,j;
     if (doc.documentElement().tagName() != "statemachine" )
     {
          ARCSLog::logCritical(logSource(doc.documentElement().lineNumber()), "\"statemachine\" tag expected.");
          return false;
     }

     QDomNodeList elements = doc.documentElement().childNodes();
     bool noerror = true ;
     bool hasTransitions = false;
     
     for ( i=0; i < elements.count() ; i++)
     {
          if (elements.at(i).isElement())
          {
               QDomElement element = elements.at(i).toElement();
               if (element.tagName() == "transitions")
               {
                    hasTransitions = true;
                    QDomNodeList transitions = element.childNodes();
                    for ( j = 0; j < transitions.count(); j++)
                    {
                         if (transitions.at(j).isElement())
                         {
                              QDomElement transition = transitions.at(j).toElement();
                              if (transition.tagName() == "transition")
                              {
                                   if ( transition.hasAttribute("source") &&
                                        transition.hasAttribute("token" ) &&
                                        transition.hasAttribute("destination") )
                                   {
                                        noerror = noerror && true;
                                        sm->addTransition(transition.attribute("source"),
                                                          transition.attribute("token"),
                                                          transition.attribute("destination"));
                                   }
                                   else
                                   {
                                        noerror = noerror && false;
                                        ARCSLog::logWarning(logSource(transition.lineNumber()), "\"transition\" tag is missing either \"source\", \"token\", or \"destination\" attribute.");
                                   }
                              }
                         }
                    }
               }

               if (element.tagName() == "first")
               {
                    if (element.hasAttribute("name"))
                         sm->setFirstSheetName(element.attribute("name"));
                    else
                         ARCSLog::logWarning(logSource(element.lineNumber()),"\"first\" tag is missing \"name\" attribute.");
               }

               if (element.tagName() == "last" )
               {
                    if (element.hasAttribute("name"))
                        sm->setLastSheetName(element.attribute("name"));
                    else
                         ARCSLog::logWarning(logSource(element.lineNumber()),"\"last\" tag is missing \"name\" attribute.");
               }
          }
     }

     if (!hasTransitions)
          ARCSLog::logWarning(logSource(doc.documentElement().lineNumber()),"statemachine without any transition.");


     return ( noerror && hasTransitions ) ;
}



bool ARCSXMLHandler::parseProcess(ARCSProcess* ap, QDomElement processNode)
{
     if (processNode.tagName() != "process")
     {
          ARCSLog::logError(logSource(processNode.lineNumber()),"\"process\" tag expected.");
          return false;
     }

     if (processNode.hasAttribute("controller"))
          if (!ap->setController(processNode.attribute("controller")))
          {
          ARCSLog::logError(logSource(processNode.lineNumber()),"undefined controller: "+ processNode.attribute("controller"));
          return false;
     }

     QDomNodeList lst = processNode.elementsByTagName("sheet");
     if (lst.count() <= 0)
     {
          ARCSLog::logError(logSource(processNode.lineNumber()),"\"process\" tag does not contain any \"sheets\" tag.");
          return false;
     }

     for (int i=0; i< lst.count(); i++)
     {
          ARCSSheet sh(ap->getContext());
          if (!parseSheet(&sh,lst.at(i).toElement()))
          {
               ARCSLog::logError(logSource(lst.at(i).lineNumber()), "failed to parse sheet.");
               return false;
          }
          //! \todo See how to prevent sheet name duplication.
          ap->addSheet(sh.getName(),sh);
     }

     return true;
}


bool ARCSXMLHandler::parseProfile(ARCSContext* ctx)
{
     QDomElement app = doc.documentElement();

     if (app.tagName() != "profile")
     {
          ARCSLog::logError(logSource(doc.documentElement().lineNumber()),"\"profile\" tag expected.");
          return false;
     }

     parseContextElement(app, ctx, "constant", &ARCSContext::modifyConstant );
     return true;
}



bool ARCSXMLHandler::parseApplication(ARCSApplicationComponent* aac)
{
     QDomElement app = doc.documentElement();

     if (app.tagName() != "application")
     {
          ARCSLog::logError(logSource(doc.documentElement().lineNumber()),"\"application\" tag expected.");
          return false;
     }

     if (app.hasAttribute("mode") && aac->getCurrentMode() == ARCS::ARCS_APP_NONE)
     {
	  QString mode = app.attribute("mode").toLower();

	  if (mode == "base")
            aac->setCurrentMode(ARCS::ARCS_APP_BASE);
	  else
	       if (mode == "thread")
              aac->setCurrentMode(ARCS::ARCS_APP_THREAD);
	       else
		    if (mode == "threadevent")
                aac->setCurrentMode(ARCS::ARCS_APP_THREADEVENT);
		    else
			 if (mode == "gui")
                     aac->setCurrentMode(ARCS::ARCS_APP_GUI);
			 else
                     aac->setCurrentMode(ARCS::ARCS_APP_EVENT);
     }

     ARCSProcess* process0 = new ARCSProcess();
     process0->setApplicationMode(aac->getCurrentMode());
     ARCSContext* context = aac->getContext();

     if (!parseContext(context))
     {
          ARCSLog::logError(logSource(),"could not parse context.");
          return false;
     }

     QDomNodeList lst = doc.elementsByTagName(QString("process"));
     if (lst.count() <= 0)
     {
          ARCSLog::logError(logSource(),"undefined \"process\" tag for application.");
          return false;
     }

     ARCSProcess* process;

     for (int i=0; i < lst.count(); i++)
     {
          if (i != 0)
               process = new ARCSProcess();
          else
               process = process0;
          process->setContext(context);
          if (!parseProcess(process, lst.at(i).toElement()))
          {
               ARCSLog::logError(logSource(lst.at(i).lineNumber()),"could not parse \"process\" tag.");
               return false;
          }
          aac->addProcess(process);
     }

     return true;
}



bool ARCSXMLHandler::parseCompositeComponent(ARCSCompositeComponent* acc)
{
     QDomElement cmp = doc.documentElement();

     if (cmp.tagName() !=  "composite")
     {
          ARCSLog::logError(logSource(doc.documentElement().lineNumber()),"\"composite\" tag expected.");
          return false;
     }

     ARCSContext* context= new ARCSContext();
     if (!parseContext(context))
     {
          ARCSLog::logError(logSource(),"could not parse context.");
          return false;
     }

     QDomNodeList lst = doc.elementsByTagName(QString("sheet"));

     if (lst.count() != 1)
     {
         ARCSLog::logError(logSource(lst.at(1).lineNumber()),"wrong number of sheets in composite component.");
          return false;
     }

     QDomElement sheetNode = lst.at(0).toElement();
     ARCSSheet sheet(context);

     parseSheet(&sheet, sheetNode);
     acc->setSheet(sheet);

     QDomNodeList _lst2 = doc.elementsByTagName(QString("interface"));
     if (_lst2.count() != 1)
     {
         ARCSLog::logWarning(logSource(_lst2.at(1).lineNumber()),"wrong number of interfaces in composite component.");
          return true;
     }

     QDomElement ifNode = _lst2.at(0).toElement();

     if (!parseInterface(acc, ifNode))
     {
         ARCSLog::logError(logSource(ifNode.lineNumber()),"could not parse \"interface\" tag.");
          return false;
     }
     return true;
}


bool ARCSXMLHandler::parseInterface(ARCSAbstractComponent* aac, QDomNode node)
{
     QDomElement root;
     int i ,j ;

     if (node.isNull())
	  root = doc.documentElement();
     else
	  if (node.isElement())
	       root = node.toElement();
     
     if (root.tagName() != "interface")
     {
         ARCSLog::logError(logSource(root.lineNumber()),"\"interface\" tag expected.");
	  return false;
  }

     QStringList signalAliases ;
     QStringList slotAliases;
     QStringList sourceComponents;
     QStringList destinationComponents ;
     QStringList signalMethods ;
     QStringList slotMethods; 

     QDomNodeList dnl = root.childNodes();
     for (i = 0; i < dnl.count(); i++)
     {
	  QDomElement elt = dnl.at(i).toElement();
	  if (elt.tagName() == "signals")
	       parseMethodList(elt, signalAliases, sourceComponents, signalMethods);

	  if (elt.tagName() == "slots")
	       parseMethodList(elt, slotAliases, destinationComponents, slotMethods);
     }


     ARCSCompositeComponent* acc = dynamic_cast<ARCSCompositeComponent*>(aac);

     if (acc != 0)
     {
	  if (signalAliases.count() != sourceComponents.count() )
          {
              ARCSLog::logError(logSource(),"signal count mismatch.");
	       return false;
       }

	  if (slotAliases.count() != destinationComponents.count())
       {
              ARCSLog::logError(logSource(),"slot count mismatch.");
	       return false;
       }

       for (j=0; j < signalAliases.count(); j++)
            acc->addProxySignal(signalAliases.at(j), sourceComponents.at(j), signalMethods.at(j));

	  for (j=0; j < slotAliases.count(); j++)
	       acc->addProxySlot(slotAliases.at(j), destinationComponents.at(j), slotMethods.at(j));

     }
     else
	  return false;

     return true;
}



void ARCSXMLHandler::parseMethodList(QDomElement elt, QStringList & aliases, QStringList & components, QStringList & methods)
{
     QDomNodeList list = elt.childNodes();

     aliases.clear();
     components.clear();
     methods.clear();     
     
     for(int i = 0; i< list.count(); i++)
     {
          if (list.at(i).isElement())
          {
               QDomElement element = list.at(i).toElement();
               if (element.tagName() == "method")
               {
                    if (element.hasAttribute("alias"))
                    {
                         aliases << element.attribute("alias");
                         if (element.hasAttribute("component") && element.hasAttribute("method"))
                         {
                              components << element.attribute("component");
                              methods << element.attribute("method");
                         }
                         else
                             ARCSLog::logWarning(logSource(element.lineNumber()),"pure method detected.");
                    }
                    else
                    {
                        ARCSLog::logWarning(logSource(element.lineNumber()),"\"method\" tag is missing \"alias\" attribute.");
                    }
               }
          }
     }
}

void ARCSXMLHandler::parseContextElement(QDomElement elt, ARCSContext* ctx, QString tagName, ARCSContext::AddPoolMethod method)
{
     if (! elt.hasChildNodes())
     {
          // uniquely used for constants.
         ARCSLog::logWarning(logSource(elt.lineNumber()),"empty \"constants\" part");
          return ;
     }

     QDomNodeList list = elt.childNodes();

     for (int i=0; i < list.count(); i++)
     {
          if (list.at(i).isElement())
          {
               QDomElement e = list.at(i).toElement();
               if (e.tagName() == tagName)
               {
                    if (!e.hasAttribute("id") || !e.hasAttribute("type"))
                    {
                        ARCSLog::logWarning(logSource(e.lineNumber()),"\"constant\" tag is missing either \"id\" or \"type\" attributes.");
                         return ;
                    }
               }

               QString id = e.attribute("id");
               QString type = e.attribute("type");
               QString contents = nodeInternalsToText(e);
               if (!(ctx->*method)(id, type, contents))
                   ARCSLog::logWarning(logSource(e.lineNumber()),"could not parse \"constant\" tag.");
          }
     }
}


bool ARCSXMLHandler::parseComponents(QDomElement elt, ARCSContext* ctx)
{
     if (! elt.hasChildNodes())
     {
         ARCSLog::logError(logSource(elt.lineNumber()),"empty \"component\" list.");
          return false;
     }

     QDomNodeList list = elt.childNodes();

     for (int i=0; i < list.count(); i++)
     {
       if (list.at(i).isElement())
       {
            QDomElement e = list.at(i).toElement();
            if (e.tagName() == "component")
            {
              if (!e.hasAttribute("id") || !e.hasAttribute("type"))
                 {

                  ARCSLog::logWarning(logSource(e.lineNumber()),
                                      "\"component\" tag is missing either \"id\" of \"type\" attributes.");
                return false;
           }
              QString id = e.attribute("id");
              QString type = e.attribute("type");
              QString filename = solveRelativePath(e.attribute("file"));
              if (! filename.isEmpty())
              {
                   ARCSAbstractComponent* aac = ctx->createComponent(id,type);
                   if (!aac)
                   {
                       ARCSLog::logError(logSource(e.lineNumber()),
                                         "could not create component with id "+id +" and type "+type) ;
                        return false;
                   }
                   aac->setProperty("filename",filename);
                   if (QFileInfo(filename).exists())
                   {
                        if (!aac->loadFile(filename))
                            ARCSLog::logWarning(logSource(e.lineNumber()),
                                                "failed to parse component.");
                   }
                   else
                   {
                       ARCSLog::logError(logSource(e.lineNumber()),
                                         "component description "+ filename + " not found.");
                   }
                   //pathStack.popPath();
              }
              else
              {
                    QString contents = nodeInternalsToText(e);
                    if (!ctx->addComponent(id,type,contents))
                        ARCSLog::logWarning(logSource(e.lineNumber()),
                                            "could not parse component.");
              }
            }
       }
     }
     return true;
}

void ARCSXMLHandler::computeRelativePaths(QString basePath)
{
    QFileInfo fi1(basePath);
    QDir dir;

    if (fi1.isDir())
        dir = QDir();
    else
        dir = fi1.absoluteDir();

    // here fileAttributes should have absolute path value.
    for (int i=0; i < fileAttributes.count(); i++)
    {
        fileAttributes[i].setValue(
                    dir.relativeFilePath(
                        QFileInfo(fileAttributes[i].value()).absoluteFilePath()
                        )
                    );
    }
}

QString ARCSXMLHandler::solveRelativePath(QString path)
{
    if (path.isEmpty())
        return QString::null;

    if (fileName.isEmpty())
        return path;

    if (QFileInfo(path).isAbsolute())
        return path;

    QFileInfo fi1(fileName);
    QDir dir;

    if (fi1.isDir())
        dir = QDir();
    else
        dir = fi1.absoluteDir();

    return QDir::cleanPath(dir.absoluteFilePath(path));
}


void ARCSXMLHandler::storeSheet(ARCSSheet* sheet,QDomElement father, QString id)
{
     int i,j;

     bool bPres,bPosts, bCons, bCleanups;
     bPres = bPosts = bCons = bCleanups = false;

     QDomElement sheetNode = doc.createElement("sheet");
     QDomElement pres = doc.createElement("preconnections");
     QDomElement posts = doc.createElement("postconnections");
     QDomElement cons = doc.createElement("connections");
     QDomElement cleanups = doc.createElement("cleanups");

     if (!id.isEmpty())
         sheetNode.setAttribute("id",id);

     QStringList sources;
     QStringList sgnls;
     QStringList destinations;
     QStringList slts;


     storeSheetProperties(sheet,sheetNode);
     bPres =     storeInvocations(sheet, &ARCSSheet::getPreconnects, pres);
     bPosts =    storeInvocations(sheet, &ARCSSheet::getPostconnects, posts);
     bCleanups = storeInvocations(sheet, &ARCSSheet::getCleanups, cleanups);
     
     sheet->getConnections(sources, sgnls, destinations, slts);
     bCons = sources.count() > 0;
     for (i=0; i < sources.count(); i++)
     {
	  QDomElement link = doc.createElement("link");
	  link.setAttribute("source", sources[i]);
	  link.setAttribute("destination", destinations[i]);
	  link.setAttribute("signal", sgnls[i]);
	  link.setAttribute("slot", slts[i]);
      ARCSConnection &connection = sheet->getConnection(
                  sources[i],sgnls[i],destinations[i],slts[i]
                  );
      QList<QPointF>& coords = connection.getCoordinates();
      QStringList coordStrings;
      if (coords.count() != 0)
      {
          for (j=0; j < coords.count(); j++)
              coordStrings.append(QString::number(coords[j].x()) + "," + QString::number(coords[j].y()));
          link.setAttribute("coords",coordStrings.join(" "));
      }
      cons.appendChild(link);

     }

     
     father.appendChild(sheetNode);
     if (bPres)
       sheetNode.appendChild(pres);
     if (bPosts)
       sheetNode.appendChild(posts);
     if (bCleanups)
       sheetNode.appendChild(cleanups);

     sheetNode.appendChild(cons);
}


void ARCSXMLHandler::storeSheetProperties(ARCSSheet *sheet, QDomElement root)
{
    QStringList propertyKeys = sheet->getPropertyList();
    if (propertyKeys.count() <= 0)
        return;

    QDomElement properties = doc.createElement("properties");
    root.appendChild(properties);
    for (int i=0; i<propertyKeys.count(); i++)
    {
        QDomElement property = doc.createElement("property");
        property.setAttribute("id",propertyKeys[i]);
        QDomText txt=doc.createTextNode(sheet->getProperty(propertyKeys[i]));
        properties.appendChild(property);
        property.appendChild(txt);
    }
}


bool ARCSXMLHandler::storeInvocations(ARCSSheet* sheet,
                                      ARCSSheet::GetInvocationsMethod method,
                                      QDomElement root)
{
     QStringList destinations;
     QStringList slts;
     QStringList types;
     QStringList values;
     int i;
     
     (sheet->*method)(destinations, slts, types, values);

     if (destinations.count() <= 0)
	  return false;

     for (i=0; i < destinations.count(); i++)
     {
	  QDomElement invoke = doc.createElement("invoke");
	  if (! values[i].isEmpty()) 
	  {
	       QDomText txt = doc.createTextNode(values[i]);
	       invoke.appendChild(txt);
	  }
	  invoke.setAttribute("destination", destinations[i]);
	  invoke.setAttribute("slot", slts[i]);
	  invoke.setAttribute("type", types[i]);
	  root.appendChild(invoke);
     }
     return true;
}


void ARCSXMLHandler::storeStateMachine(ARCSStateMachine* sm)
{
     type = XML_STATEMACHINE;

     QDomElement smNode = doc.createElement("statemachine");
     QDomElement trsNode = doc.createElement("transitions");
         
     doc.appendChild(smNode);
     smNode.appendChild(trsNode);

     QDomElement fstNode = doc.createElement("first");
     fstNode.setAttribute("name", sm->getFirstSheetName());
     smNode.appendChild(fstNode);


     if (!sm->getLastSheetName().isEmpty())
     {
	  QDomElement lstNode = doc.createElement("last");
	  lstNode.setAttribute("name", sm->getLastSheetName());
	  smNode.appendChild(lstNode);
     }
     

     QStringList sources;
     QStringList tokens;
     QStringList destinations;
     sm->getTransitions(sources, tokens, destinations);

     for (int i=0; i < sources.count(); i++)
     {
	  QDomElement trNode = doc.createElement("transition");
	  trNode.setAttribute("source",sources[i]);
	  trNode.setAttribute("token", tokens[i]);
	  trNode.setAttribute("destination", destinations[i]);
	  trsNode.appendChild(trNode);
     }
}


void ARCSXMLHandler::storeProfile(ARCSContext* ctx)
{
     type = XML_PROFILE ;

     QDomElement app = doc.createElement("profile");
     doc.appendChild(app);

     QStringList lst = ctx->getConstantList();

     for (int i=0; i < lst.count(); i++)
     {
       QDomElement cstNode = doc.createElement("constant");
       QString t,r;

       if (ctx->serializeConstant(lst[i],t,r))
       {
            cstNode.setAttribute("id",lst[i]);
            cstNode.setAttribute("type",t);

            QDomText cstTxt = doc.createTextNode(r);
            cstNode.appendChild(cstTxt);
            app.appendChild(cstNode);
       }
     }
}


void ARCSXMLHandler::storeContext(ARCSContext* ctx)
{
     int i;

     QDomElement ctxNode  = doc.createElement("context");
     QDomElement libsNode = doc.createElement("libraries");
     QDomElement cmpsNode = doc.createElement("components");
     QDomElement cstsNode = doc.createElement("constants");


     QStringList lst = ctx->getComponentList();

     for (i=0; i < lst.count(); i++)
     {
	  QDomElement cmpNode = doc.createElement("component");
	  ARCSAbstractComponent* aac = ctx->getComponent(lst[i]);
	  cmpNode.setAttribute("id",lst[i]);
	  cmpNode.setAttribute("type",aac->getType());
	  
      QString filename = aac->getProperty("filename").toString();
      if (filename.isEmpty() || QFileInfo(filename).isRelative())
      {
          // very dirty hack. I still do not know if it works.
          QDomDocument cmpTMP;
          if (cmpTMP.setContent(aac->toString()))
              cmpNode.appendChild(cmpTMP);
          else
              cmpNode.appendChild(doc.createTextNode(aac->toString()));
          // end of hack
      }
      else
      {
          QDomAttr fAttr = doc.createAttribute("file");
          fAttr.setValue(filename);
          //cmpNode.appendChild(fAttr);
          cmpNode.setAttributeNode(fAttr);
          fileAttributes.push_back(fAttr);
      }

	  cmpsNode.appendChild(cmpNode);
     }

     lst = ctx->getConstantList();

     for (i=0; i < lst.count(); i++)
     {
	  QDomElement cstNode = doc.createElement("constant");
	  QString t,r;
	  
	  if (ctx->serializeConstant(lst[i],t,r))
	  {
	       cstNode.setAttribute("id",lst[i]);
	       cstNode.setAttribute("type",t);
	       
	       QDomText cstTxt = doc.createTextNode(r);
	       cstNode.appendChild(cstTxt);
	       cstsNode.appendChild(cstNode);
	  }
     }

     lst = ctx->computeLibraryList();

     for (i=0; i < lst.count(); i++)
     {
         QDomElement libNode = doc.createElement("library");

         //QString relPath = pushPath?pathStack.relativePath(lst[i]):lst[i];
         //libNode.setAttribute("path",relPath);
         QDomAttr fAttr = doc.createAttribute("path");
         fAttr.setValue(lst[i]);
         //libNode.appendChild(fAttr);
         libNode.setAttributeNode(fAttr);
         fileAttributes.push_back(fAttr);
         libsNode.appendChild(libNode);
     }

     ctxNode.appendChild(libsNode);
     ctxNode.appendChild(cmpsNode);
     ctxNode.appendChild(cstsNode);
     doc.documentElement().appendChild(ctxNode);
}



void ARCSXMLHandler::storeApplication(ARCSApplicationComponent* aac)
{
     type = XML_APPLICATION ;

     QDomElement app = doc.createElement("application");
     doc.appendChild(app);

     switch(aac->getCurrentMode())
     {
     case ARCS::ARCS_APP_BASE:
         app.setAttribute("mode","base");
         break;
     case ARCS::ARCS_APP_THREAD:
         app.setAttribute("mode","thread");
         break;
     case ARCS::ARCS_APP_THREADEVENT:
         app.setAttribute("mode", "threadevent");
         break;
     case ARCS::ARCS_APP_GUI:
         app.setAttribute("mode", "gui");
         break;
     default:
         app.setAttribute("mode", "event");

     }

     storeContext(aac->getContext());
     
     QDomElement prcs = doc.createElement("processes");
     app.appendChild(prcs);
     int nProc = aac->getProcessCount();

     for (int i = 0; i < nProc; i++)
          storeProcess(aac->getProcess(i), prcs);
}


void ARCSXMLHandler::storeProcess(ARCSProcess* proc,QDomElement father)
{
     QDomElement process = doc.createElement("process");

     process.setAttribute("controller", proc->getControllerId());

     QStringList sheetNames = proc->getSheetNames();

     for(int i=0; i< sheetNames.count(); i++)
     {
          ARCSSheet& as = proc->getSheet(sheetNames[i]);
          storeSheet(&as,process,sheetNames[i]);
     }
     father.appendChild(process);
}



void ARCSXMLHandler::storeCompositeComponent(ARCSCompositeComponent* acc)
{
     type = XML_COMPOSITE;

     QDomElement composite = doc.createElement("composite");
     doc.appendChild(composite);
     
     ARCSContext ctx = acc->getContext();

     storeContext(&ctx);

     storeSheet( acc->getSheet(), composite);
     
     storeInterface( acc, composite);
}

//! \todo Not yet tested.
void ARCSXMLHandler::storeInterface(ARCSAbstractComponent* aac, QDomElement elt)
{
    int i;

    QDomElement interfaceARCS = doc.createElement("interface");
    elt.appendChild(interfaceARCS);

    ARCSCompositeComponent* acc = dynamic_cast<ARCSCompositeComponent*>(aac);
    if (acc != NULL)
    {

        QStringList signalNames;
        QStringList slotNames;
        QStringList methodNames;
        QStringList componentNames;
        acc->getProxySignals(methodNames, componentNames, signalNames);
        if (methodNames.count() == componentNames.count() &&
                methodNames.count() == signalNames.count())
        {
            QDomElement signalRoot = doc.createElement("signals");
            interfaceARCS.appendChild(signalRoot);
            for (i=0; i< methodNames.count(); i++)
            {
                QDomElement method = doc.createElement("method");
                method.setAttribute("alias",methodNames[i]);
                method.setAttribute("component", componentNames[i]);
                method.setAttribute("method",signalNames[i]);
                signalRoot.appendChild(method);
            }
        }

        acc->getProxySlots(methodNames, componentNames, slotNames);
        if (methodNames.count() == componentNames.count() &&
                methodNames.count() == slotNames.count())
        {
            QDomElement slotRoot = doc.createElement("slots");
            interfaceARCS.appendChild(slotRoot);
            for (i=0; i< methodNames.count(); i++)
            {
                QDomElement method = doc.createElement("method");
                method.setAttribute("alias",methodNames[i]);
                method.setAttribute("component", componentNames[i]);
                method.setAttribute("method",signalNames[i]);
                slotRoot.appendChild(method);
            }
        }
    }
}


void ARCSXMLHandler::reset()
{
     libraryStrings.clear();
}




QString ARCSXMLHandler::nodeInternalsToText(QDomNode node)
{
     QString res;

     QTextStream ts(&res);

     if (node.hasChildNodes())
     {
	  QDomNodeList children = node.childNodes();
	  for (int i = 0 ; i < children.count() ; i++)
	       ts << children.at(i) ;
     }

     return res;
}


