/*
  name: lib/arcsscriptcomponent.cpp

  This file is part of ARCS - Augmented Reality Component System
  (version 2-current), written by Jean-Yves Didier 
  for IBISC Laboratory (http://www.ibisc.univ-evry.fr)

  Copyright (C) 2013  Universit d'Evry-Val d'Essonne

  This program is free software: you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program.  If not, see <http://www.gnu.org/licenses/>.


  Please send bugreports  with examples or suggestions to
  jean-yves.didier__at__ibisc.univ-evry.fr
*/


#include <arcs/arcsscriptcomponent.h>
#include <QRegExp>
#include <iostream>

ARCSScriptQObjectProxy::ARCSScriptQObjectProxy(QObject* parent) : QObject(parent)
{
     engine = new QScriptEngine(this);
     engine->globalObject().setProperty("callSignal", engine->newFunction(ARCSScriptQObjectProxy::callSignal));

}


int ARCSScriptQObjectProxy::paramCount(QString s)
{
     if (s.trimmed().isEmpty())
          return 0;
     QStringList tmp = s.trimmed().split(',');
     return tmp.count();
}

int ARCSScriptQObjectProxy::qt_metacall(QMetaObject::Call call, int id, void** arguments)
{
     id = QObject::qt_metacall(call,id, arguments);

     if (id == -1 || call != QMetaObject::InvokeMetaMethod)
          return -1;


     if (id < slotList.count())
     {
          QScriptValue mySlot = engine->globalObject().property(slotList.at(id));
          QScriptValueList valueList;
          for (int i=0; i < paramSlotCount.at(id); i++)
          {
               // pb : il faut convertir en fonction du type.
               int type = (paramSlotType.at(id)).at(i);
               valueList << engine->newVariant(QVariant(type, arguments[i+1]));
          }
          QScriptValue result = mySlot.call(engine->globalObject(), valueList);
          return -1;
     }

     id -= slotList.count();


     if (id < 0)
          return -1;

     QMetaObject::activate(this, metaObject(), id + metaObject()->methodCount()+slotList.count(), arguments);

     return -1;
}


int ARCSScriptQObjectProxy::cleanSignalConnect(QString sigName, QString sltName, QString /*objectName*/, QString /*actualSlot*/)
{
     sigName = sigName.section("(",0,0);
     if (!signalList.contains(sigName))
     {
          std::cerr << "[Script] Cannot connect signal " << qPrintable(sigName) << std::endl;
          return -1;
     }


     if (signalCollection.contains(sigName))
     {
          QStringList sl = signalCollection[sigName];
          sl.removeAll(sltName);
          signalCollection[sigName] = sl;
     }


     int res = signalList.indexOf(sigName);



     return metaObject()->methodCount() + slotList.count() + res;
}

int ARCSScriptQObjectProxy::cleanSlotConnect(QString /*sigName*/,QString sltName, QString /*objectName*/, QString /*actualSignal*/)
{
     sltName=sltName.section("(",0,0);
     if (! slotList.contains(sltName))
     {
          std::cerr << "[Script] Cannot connect slot " << qPrintable(sltName) << std::endl;
          return -1;
     }

     int res = slotList.indexOf(sltName);
     return metaObject()->methodCount() + res;
}


int ARCSScriptQObjectProxy::prepareSignalConnect(QString sigName, QString sltName, QString /*objectName*/, QString /*actualSlot*/,bool /*simulate*/)
{
     sigName = sigName.section("(",0,0);
     if (!signalList.contains(sigName))
     {
          std::cerr << "[Script] Cannot connect signal " << qPrintable(sigName) << std::endl;
          return -1;
     }

     int res = signalList.indexOf(sigName);

     //std::cout << "sltName to connect to signal :" << qPrintable(sltName) << std::endl;


     // prparation des types.
     int numParams = paramCount(sltName.section("(",1,1).section(")", 0,0));
     QStringList paramTypes = sltName.section("(",1,1).section(")", 0,0).split(",");

     // premier test : si la liste est vide : on retourne rien
     if (numParams == paramSignalCount[res])
     {
          QList<int> types;

          for (int i=0; i < numParams; i++)
               types.append(QMetaType::type(qPrintable(paramTypes.at(i))));

          paramSignalType[res] = types;
     }
     return metaObject()->methodCount() + slotList.count() + res;
}

int ARCSScriptQObjectProxy::prepareSlotConnect(QString sigName, QString sltName, QString /*objectName*/, QString /*actualSignal*/, bool /*simulate*/)
{
     sltName=sltName.section("(",0,0);
     if (! slotList.contains(sltName))
     {
          std::cerr << "[Script] Cannot connect slot " << qPrintable(sltName) << std::endl;
          return -1;
     }

     int res = slotList.indexOf(sltName);

     QStringList paramTypes = sigName.section("(",1,1).section(")", 0,0).split(",");
     QList<int> types;

     for (int i=0; i < paramTypes.count(); i++)
          types.append(QMetaType::type(qPrintable(paramTypes.at(i))));

     paramSlotType[res] = types;

     return metaObject()->methodCount() + res;
}


QStringList ARCSScriptQObjectProxy::getSignalList()
{
     return signalList;
}

QStringList ARCSScriptQObjectProxy::getSlotList()
{
     return slotList;
}


void ARCSScriptQObjectProxy::createCallSignalScript()
{
     // last signal on stack :
     QString name = signalList.last();
     int num = signalList.count()-1;
     int numParams = paramSignalCount.last();

     // constitution des paramtres et des paramtres
     QStringList temp;
     QList<int> types;
     for (int i=0; i < numParams ; i++)
     {
          temp << (QString("p")+QString::number(i)) ;
          types.append(QMetaType::type("QString"));
     }
     paramSignalType[num] = types ;

     QString params = temp.join(",");

     QString signalScript = "function emit_"+ name + "(" + params + ") {\n" ;


     if (params.isEmpty())
          signalScript += "callSignal(" + QString::number(num) + ");\n";
     else
          signalScript += "callSignal(" + QString::number(num) + "," + params + ");\n";

     signalScript += "}\n\n";
     //std::cout << qPrintable(signalScript) << std::endl;

     engine->evaluate(signalScript);

     if (engine->hasUncaughtException())
          std::cerr << "[Script] Cannot evaluate script " << qPrintable(signalScript) << std::endl;

}

bool ARCSScriptQObjectProxy::setScript(QString s)
{
     QRegExp slotsSearch("function\\s+(\\S+)\\(");
     QRegExp signalSearch("emit_(\\S+)\\((.*)\\)");
     signalSearch.setMinimal(true);

     scriptString = s ;

     engine->evaluate(s);
     if (engine->hasUncaughtException())
     {
          std::cout << qPrintable(engine->uncaughtException().toString()) << std::endl ;
          return false;
     }


     int pos = 0;

     // pb : il faut compter les paramtres passs.
     while ((pos = slotsSearch.indexIn(s, pos)) != -1)
     {
          slotList << slotsSearch.cap(1);
          paramSlotCount.append(slotsSearch.cap(2).trimmed().split(',').count());
          paramSlotType.append(QList<int>());
          pos += slotsSearch.matchedLength();
          std::cout << "[Script] Found one slot " << qPrintable( slotList.last()) << std::endl;
     }

     pos = 0;
     while ((pos = signalSearch.indexIn(s, pos)) != -1)
     {
          // pb plusieurs missions pourraient avoir lieu.
          // comptons sur un seul emit .
          // la rgle devra tre prcise dans le chapitre sur les scripts.
          if (!signalList.contains(signalSearch.cap(1)))
          {
               signalList << signalSearch.cap(1);
               //std::cout << "signalSearch.cap(2)" << qPrintable(signalSearch.cap(2)) << std::endl;
               paramSignalCount.append(paramCount(signalSearch.cap(2)));
               paramSignalType.append(QList<int>());
               createCallSignalScript();
               pos += signalSearch.matchedLength();
               std::cout << "[Script] Found one signal " << qPrintable( signalList.last()) << std::endl;
          }
          else
          {
               std::cerr << "[Script] Signal " << qPrintable(signalSearch.cap(1)) << " was already found." << std::endl;
          }
     }

     return true;
}

QScriptValue ARCSScriptQObjectProxy::callSignal(QScriptContext* ctx, QScriptEngine* engine)
{
     //1 prparer les donnes
     int nbargs = ctx->argumentCount();
     void * args[nbargs+1];
     args[0] = 0 ;

     int id = ctx->argument(0).toInt32();

     ARCSScriptQObjectProxy* proxy = (ARCSScriptQObjectProxy*)engine->parent();
     QVariant var[nbargs];

     if (proxy->paramSignalType.at(id).count() < nbargs -1 )
     {
          std::cout << "[Script] Trouble : " << proxy->paramSignalType.at(id).count()
                    << " params available for " <<  nbargs -1 << "params required." << std::endl;
          return QScriptValue();

     }

     for (int i = 1; i < nbargs; i++)
     {
          var[i] = ctx->argument(i).toVariant();
          var[i].convert((QVariant::Type)proxy->paramSignalType.at(id).at(i-1));
          args[i] = var[i].data() ;
     }

     //2 lancer l'appel  l'aide de qt_metacall() ou qt_activate()
     proxy->qt_metacall( QMetaObject::InvokeMetaMethod ,id + proxy->metaObject()->methodCount() +  proxy->slotList.count(),args);
      return QScriptValue();
}




QString ARCSScriptQObjectProxy::computeParamList(QString slt)
{
     if (slt.section('(',1,1).section(')',0,0).isEmpty())
          return QString();

     int nbParams = slt.count(',') +  1 ;


     QString res;

     for (int  i = 0; i < nbParams; i++)
          res += "p" + QString::number(i);
     return res;
}





/*******************************************************************
  * ARCSScriptComponent
  *******************************************************************/



ARCSScriptComponent::ARCSScriptComponent() //: engine()
{
     setType("Script");
     proxy = new ARCSScriptQObjectProxy();

}


ARCSScriptComponent::~ARCSScriptComponent()
{
}

bool ARCSScriptComponent::parseString(QString s)
{
     scriptString = s;
     return proxy->setScript(s);
}


void ARCSScriptComponent::getProxySlot (QString slot, ObjectList &obj, QStringList &proxySlot)
{
     obj << proxy ;
     proxySlot << slot ;
}


void ARCSScriptComponent::getProxySignal (QString signal, ObjectList &obj, QStringList &proxySignal)
{
     obj << proxy ;
     proxySignal << signal;
}


