/*
  name: include/arcs/arcsinit.h

  This file is part of ARCS - Augmented Reality Component System
  (version 2-current), written by Jean-Yves Didier 
  for IBISC Laboratory (http://www.ibisc.univ-evry.fr)

  Copyright (C) 2013  Universit d'Evry-Val d'Essonne

  This program is free software: you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program.  If not, see <http://www.gnu.org/licenses/>.


  Please send bugreports  with examples or suggestions to
  jean-yves.didier__at__ibisc.univ-evry.fr
*/


#ifndef __ARCSINIT_H__
#define __ARCSINIT_H__

#include <arcs/arcsabstractcomponent.h>
#include <QBitArray>
#include <arcs/arcsdll.h>

class ARCSFactory;

//! Desribes an initialization over a component.
/*! This class represents an initialization over a component.
 * It also handles and performs initializations on the component.
 * (Here we should more properly talk of invocations)
 * \author Jean-Yves Didier
 * \date December, the 5th, 2008
 * \ingroup lib
 */
class DLL_POINT ARCSInit
{
public:
     ARCSInit(); //!< Default constructor
     ARCSInit(const ARCSInit& i); //!< Copy constructor
/*! Invocation constructor
 * 
 *  An invocation is performed on a slot of a particular component.
 * \param d Component upon which an invocation will be performed.
 * \param slt slot name through which invocation will be performed.
 * \param v value passed during invocation.
 */
     ARCSInit(ARCSAbstractComponent* d, QString slt, QVariant v);
     ~ARCSInit(); //!< Destructor
     
     /*! Sets the destination component on which invocation will be performed */
     void setDestination(ARCSAbstractComponent* d) { destination = d; }
     /*! Sets the slot through which an invocation is performed */
     void setSlotName(QString slt) { slotName = slt; }
     /*! Sets the value passed at invocation */
     void setValue(QVariant v) { value = v; }
     
     /*! Returns the invocation's destination component */
     ARCSAbstractComponent* getDestination() { return destination; }

     QString getDestinationName() { return destination->getProperty("id").toString(); }
     /*! Returns the invocation's slot name */
     QString getSlotName()  { return slotName; }
     /*! Returns the invocation's value */
     QVariant getValue() { return value; }

     /*! Returns the type of the invocation's parameter */
     QString getValueType();
     /*! Returns a string representation of the invocation's parameter */
     QString getValueString();

     /*! Peforms actual invocation.
      * init() is in three falvours, one for invocations without parameters (currently this one),
      * one for invocations with one standard parameter,
      * one for invocations with a component passed as a parameter.
      * \return <tt>true</tt> if succeeded, <tt>false</tt> otherwise.
      */
     bool init();
     /*! Peforms actual invocation.
      * init() is in three falvours, one for invocations without parameters,
      * one for invocations with one standard parameter (currently this one),
      * one for invocations with a component passed as a parameter.
      * \return <tt>true</tt> if succeeded, <tt>false</tt> otherwise.
      */
     bool init(QVariant var);
     /*! Peforms actual invocation.
      * init() is in three falvours, one for invocations without parameters,
      * one for invocations with one standard parameter,
      * one for invocations with a component passed as a parameter (currently this one).
      * \return <tt>true</tt> if succeeded, <tt>false</tt> otherwise.
      */
     bool init(ARCSAbstractComponent* cmp);


     /*! Tells wether the invocation parameter is extracted from a constant or not */
     bool isConstant() { return constant;}
     /*! Tells wether the invocation parameter is a component name or not */
     bool isComponent() { return component; }

     bool isComplete() { return destination && !slotName.isEmpty() ;}

//! \todo solve the problem of QVariant copy

     /*! Comparison operator */
     bool operator==( ARCSInit i)
     {
      return ((destination == i.destination) && (slotName == i.slotName) && (getValueString() == i.getValueString()) );
     }

     static ARCSInit null;

private:
     ARCSAbstractComponent* destination;
     QString slotName;
     QVariant value;
     static ARCSFactory* factory;
     bool constant;
     bool component ;

};

#endif //__ARCSINIT_H__
