/*
  name: tools/pkg/arcspackagemanager.cpp

  This file is part of ARCS - Augmented Reality Component System
  (version 2-current), written by Jean-Yves Didier 
  for IBISC Laboratory (http://www.ibisc.univ-evry.fr)

  Copyright (C) 2013  Universit d'Evry-Val d'Essonne

  This program is free software: you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program.  If not, see <http://www.gnu.org/licenses/>.


  Please send bugreports  with examples or suggestions to
  jean-yves.didier__at__ibisc.univ-evry.fr
*/


#include "arcspackagemanager.h"
#include "arcsdownloader.h"
#include <QSettings>
#include <QDir>
#include <QFile>
#include <QFileInfo>
#include <QTextStream>
#include <QSettings>
#include <QProcess>
#include <iostream>
#include <QDomDocument>




ARCSPackageManager::ARCSPackageManager()
{
    ad = new ARCSDownloader();
    hS["repositoryPath"]="";
    hS["repositoryUrl"]="";
    hS["repositoryFile"]="";
    hS["sourcePath"]="";
    hS["archiveExtension"]="";
    hS["compressCommand"]="";
    hS["unCompressCommand"]="";
}

ARCSPackageManager::~ARCSPackageManager()
{
    delete ad;
}

bool ARCSPackageManager::checkSettings()
{
    fetchSetting("repositoryPath");
    fetchSetting("repositoryUrl");
    fetchSetting("repositoryFile","REPOSITORY");
    fetchSetting("archiveExtension",".zip");
    fetchSetting("compressCommand","zip -r %a %p");
    fetchSetting("unCompressCommand","unzip -o %a");
    if (!settings.contains("pkg/sourcePath"))
    {
        char* buildpath = getenv("ARCSBUILDPATH");
        if (!buildpath)
        {
            std::cerr << "You must define a ARCSBUILPATH environment variable" << std::endl;
            return false;
        }
        hS["sourcePath"] = buildpath;
    }
    else
    {
        fetchSetting("sourcePath");
    }
    return true;
}


void ARCSPackageManager::displaySettings()
{
    std::cout << "Current settings" << std::endl;
    QHashIterator<QString,QString> i(hS);
    while(i.hasNext())
    {
        i.next();
        displaySetting(i.key());
    }
}


void ARCSPackageManager::storeSettings()
{
    QHashIterator<QString,QString> i(hS);
    while(i.hasNext())
    {
        i.next();
        storeSetting(i.key());
    }
}

//! \todo when compressing the archive, scan the source repository to check related files
void ARCSPackageManager::buildRepository()
{
    // first parse the REPOSITORY file
    QFile data(hS["sourcePath"] + QDir::separator() + hS["repositoryFile"]);
    parseRepositoryFile(data);

    // copy the REPOSITORY file in repository path
    data.copy(hS["repositoryPath"] + QDir::separator() + hS["repositoryFile"]);

    QDir::setCurrent(hS["sourcePath"]);
    // then compress all files and put them in repositoryPath
    for (int i=0; i < libs.count() ; i++)
    {
        // put a new function here to parse and add files is required
        if (QFileInfo(libPaths[i]).exists())
        {
            QString command = hS["compressCommand"];
            command.replace("%a",hS["repositoryPath"]+QDir::separator()+libs[i]+hS["archiveExtension"])
                    .replace("%p",libPaths[i]);
            std::cout << "Performing command: " << qPrintable(command) << std::endl;
            QProcess::execute(command);

            QStringList relatedFiles = getRelatedFiles(hS["sourcePath"]+QDir::separator()+libPaths[i]);
            for (int j=0; j < relatedFiles.count(); j++)
            {
                if (QFileInfo(relatedFiles[j]).exists())
                {
                    QString command = hS["compressCommand"];
                    command.replace("%a",hS["repositoryPath"]+QDir::separator()+libs[i]+hS["archiveExtension"])
                            .replace("%p",relatedFiles[j]);
                    std::cout << "Performing command: " << qPrintable(command) << std::endl;
                    QProcess::execute(command);
                }
            }
        }
        else
            std::cerr << "Library '" << qPrintable(libs[i]) << "': path '"
                      << qPrintable(libPaths[i]) << "' does not exist." << std::endl;
    }
}

void ARCSPackageManager::installAll()
{
    QDir::setCurrent(hS["sourcePath"]);
    ad->download(hS["repositoryUrl"]+"/"+hS["repositoryFile"]);

    if (!QFile::exists(hS["repositoryFile"]))
        return;

    QFile data(hS["repositoryFile"]);
    parseRepositoryFile(data);

    for (int i=0; i< libs.length(); i++)
    {
        installLibrary(libs[i]);
    }
}



void ARCSPackageManager::installLibrary(QString libraryName)
{
    QDir::setCurrent(hS["sourcePath"]);
    QString fullName = libraryName+hS["archiveExtension"];
    ad->download(hS["repositoryUrl"]+"/"+fullName);

    if(QFile::exists(fullName))
    {
        QString command = hS["unCompressCommand"];
        command.replace("%a",fullName);
        if(!QProcess::execute(command))
            QFile::remove(fullName);
        else
        {
            std::cerr << "Failed to perform command: " << qPrintable(command) << std::endl;
        }
    }

}


void ARCSPackageManager::parseRepositoryFile(QFile& data )
{
    libs.clear();
    libPaths.clear();

    if (!data.open(QFile::ReadOnly))
    {
        std::cerr << "Failed to open repository file: "
                  << qPrintable(hS["sourcePath"] + QDir::separator() + hS["repositoryFile"]) << std::endl;
        return;
    }

    std::cout << "Parsing repository file: "
              << qPrintable(hS["repositoryFile"]) << std::endl;

    QTextStream repos(&data);
    QString line;
    do
    {
        line = repos.readLine();
        QStringList sl=line.split(';');

        if (sl.count() >= 2 && !sl[0].startsWith("#"))
        {
            std::cout << "Registering library " << qPrintable(sl[0]) << std::endl;
            libs << sl[0];
            libPaths << sl[1];
        }
    } while(!line.isNull());
    data.close();
}

QStringList ARCSPackageManager::getRelatedFiles(QString libraryPath)
{
    QDir dir(libraryPath);
    if (!dir.exists())
        return QStringList();

    QStringList alxFiles = dir.entryList(QStringList(QString("*.alx")),QDir::Files);
    QStringList res;

    for (int i=0; i<alxFiles.count() ; i++)
    {
        QDomDocument doc;
        QFile alxFile(dir.absolutePath()+QDir::separator()+alxFiles[i]);
        if (alxFile.open(QIODevice::ReadOnly))
        {
            if (doc.setContent(&alxFile))
            {
                QDomElement root = doc.documentElement();
                QDomNodeList dnl = root.elementsByTagName("related");

                for (int j=0; j< dnl.count(); j++)
                {
                    QDomElement elt = dnl.at(j).toElement();
                    QString s = elt.attribute("path");
                    if (!s.isEmpty())
                        res << s;
                }
            }
        }
    }
    return res;
}
