/*
  name: tools/engine/main.cpp

  This file is part of ARCS - Augmented Reality Component System
  (version 2-current), written by Jean-Yves Didier 
  for IBISC Laboratory (http://www.ibisc.univ-evry.fr)

  Copyright (C) 2013  Universit d'Evry-Val d'Essonne

  This program is free software: you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program.  If not, see <http://www.gnu.org/licenses/>.


  Please send bugreports  with examples or suggestions to
  jean-yves.didier__at__ibisc.univ-evry.fr
*/


#include <arcs/arcsapplicationcomponent.h>
#include <arcs/arcsxmlhandler.h>
#include <arcs/arcslog.h>


/*! 
 * \mainpage ARCS next generation API
 *
 *
 * <h2> Where to start</h2>
 * You can see some important classes like:
 * <ul>
 * <li>ARCSFactory, a singleton handling all component libraries;</li>
 * <li>ARCSAbstractComponent, the class describing component behaviours;</li>
 * <li>ARCSAbstractFamily, the class used to describe exogenous component systems;</li>
 * <li>ARCSTypeFactoryTemplate, the template used to make it possible to recognize new data types in the engine;</li>
 * <li>ARCSXMLHandler, the class that parses most XML files used in ARCS;</li>
 * <li>and manpages for \ref arcsengine "arcsengine", \ref arcslibmaker "arcslibmaker",
 * \ref arcslibdump "arcslibdump", \ref arcspkg "arcpkg", \ref arcsbuilder "arcsbuilder", \ref arcs1to2 "arc1to2".</li>
 * </ul>
 *
 * <h2>Developping new extensions to ARCS</h2>
 * The ARCS engine supports the following extensions:
 * <ul>
 * <li>Definitions of new types (see arcsinternaltypes.h for an example and ARCSTypeFactoryTemplate for theory);</li>
 * <li>Definitions of arrays of new types (see arcsarray.h); </li>
 * <li>Definitions of new components (see classes Loop, Viewer, DisplayInt, StackExploder as examples);</li>
 * <li>Definitions of qt components with very specific behaviours (see classes ARCSQDynamicObject, ARCSGeneralLogger); </li>
 * <li>Definitions of family of components to integrate other component systems (see classes ARCSInternalFamily as samples) ;</li>
 * </ul>
 *
 * <h2>Embedding the ARCS Engine</h2>
 * To see how to embed the ARCS engine into your applications have a look at tools/engine/main.cpp
 *
 * <h2>Where to find additional documentation</h2>
 * You can consult the website http://arcs.ibisc.univ-evry.fr/
 *
 */


/*! \page arcsengine arcsengine manpage
 *<H2>NAME</H2>
 *<B>arcsengine </B>
 *: launches an XML application description using ARCS engine.
 *<H2>SYNOPSIS</H2>
 *<B>arcsengine </B>
 *[OPTION]... XMLFILE
 *<H2>DESCRIPTION</H2>
 *<B>arcsengine </B>
 *parses and runs an application described using an XML format.
 *It can run the application into several modes according to the behaviour that
 *is expected.
 *<P>
 *<H2>OPTIONS</H2>
 *<H3>Help and logging</H3>
 *<DL COMPACT>
 *<DT>
 *<B>-h, --help</B>
 *<DD>
 *<DD>prints a short help summary.
 *<DT>
 *<B>-l</B>
 *[0-2]
 *<DD>
 *<DD>defines output log mode.
 *<BR>
 *<B>0</B>
 *: do not display logs.
 *<BR>
 *<B>1</B>
 *: display colorized logs in console.
 *<BR>
 *<B>2 </B>
 *: display colorized logs for an HTML parser.
 *</DL>
 *<H3>Overriding application behaviour</H3>
 *<DL COMPACT>
 *<DT>
 *<B>-b, --mode-base</B>
 *<DD>
 *<DD>simple loop based applications
 *<DT>
 *<B>-e, --mode-event</B>
 *<DD>
 *<DD>event loop based console applications. You must use it if you want
 *asynchronous connections.
 *<DT>
 *<B>-g, --mode-gui</B>
 *<DD>
 *<DD>event loop based GUI applications. It is required if you want to start
 *applications using Qt Widgets.
 *<DT>
 *<B>-t, --mode-thread</B>
 *<DD>
 *<DD>threaded application, that is to say the application will start in a
 *secondary thread instead of the main thread. This is mainly for testing
 *purposes.
 *<DT>
 *<B>-te, --mode-thread-event</B>
 *<DD>
 *<DD>threaded application, with event loops. It is also for testing purposes.
 *<P>
 *</DL>
 *<H3>Profiles and constants</H3>
 *<DL COMPACT>
 *<DT>
 *<B>-d, --define <I>vars...</I></B>
 *<DD>
 *redefines constants in the XML file.
 *<B>var...</B>
 *is a list of constants assignations separated by colons.
 *<DT>
 *<B>-p, --profile <I>file</I></B>
 *<DD>
 *sets a profile. Constants are assignated using the contents of
 *<B>file.</B>
 *<DT>
 *<B>-o, --profile-out <I>file</I></B>
 *<DD>
 *defines a
 *<B>file</B>
 *where to output modified constants
 *</DL>
 *<H2>EXAMPLES</H2>
 *<DL COMPACT>
 *<DT>
 *<B>arcsengine loop.xml</B>
 *<DD>
 *<DD>launches the application named loop.xml
 *<DT>
 *<B>arcsengine -e loop.xml</B>
 *<DD>
 *<DD>launches the application loop.xml and overrides its current running mode by
 *using event loops.
 *<DT>
 *<B>arcsengine -d iterations=10 loop.xml</B>
 *<DD>
 *<DD>launches the application loop.xml and redefines the value of the constant
 *named &quot;iterations&quot;.
 *</DL>
 *<H2>AUTHOR</H2>
 *Jean-Yves Didier
 */

/*! \file main.cpp
 * 
 * Sample file showing how to instanciate and launch ARCS Applications.
 * It is also the program launching ARCS applications provided by default with the 
 * ARCS framework. 
 * 
 * 
 *  \author Jean-Yves Didier
 *  \date April, the 7th, 2009
 *  \ingroup sample
 */

void usage(char* name)
{
    std::cout << name << " [OPTION]...  [XML_FILE]..." << std::endl 
        << std::endl;
    std::cout << "  -h,  --help             : print this help" << std::endl << std::endl;
    std::cout << "Overriding application mode : " << std::endl << std::endl;
    std::cout << "  -b,  --mode-base        : simple loop based applications." 
        << std::endl;
    std::cout << "  -e,  --mode-event       : event loop based console applications."
        << std::endl;
    std::cout << "  -g,  --mode-gui         : event loop based GUI applications." 
        << std::endl;
    std::cout << "  -t,  --mode-thread      : threaded application." << std::endl;
    std::cout << "  -te, --mode-thread-event: threaded event based application." << std::endl <<std::endl;
    std::cout << "Defining options:" << std::endl;
    std::cout << "  -d, --define vars       : define constants" << std::endl; 
    std::cout << "  -p, --profile file      : define a profile" << std::endl;
    std::cout << "  -o, --profile-out       : define a file where to dump profile" << std::endl;
    std::cout << "  -l [0-2]                : define output log mode - 0, none; 1, console (default); 2, html" << std::endl;
}


void parseOptions(int argc, char* argv[], ARCS::ARCSAppFlag & flag, QString & constantList, QString & xmlfile, QString & profile, QString& profileOut)
{
     int i=1;
     flag = ARCS::ARCS_APP_NONE;

     for(i=1; i < argc; i++)
     {
          QString current(argv[i]);

          if (current == "-h" || current == "--help")
               usage(argv[0]);
          else
               if (current == "-b" || current == "--mode-base")
                    flag = ARCS::ARCS_APP_BASE;
          else
               if (current == "-e" || current == "--mode-event")
                    flag = ARCS::ARCS_APP_EVENT;
          else
               if (current == "-g" || current == "--mode-gui")
                    flag = ARCS::ARCS_APP_GUI;
          else
               if (current == "-t" || current == "--mode-thread")
                    flag = ARCS::ARCS_APP_THREAD;
          else
               if (current == "-te" || current == "--mode-thread-event")
                    flag = ARCS::ARCS_APP_THREADEVENT;
          else
               if (current == "-l")
               {
                   if (i+1 < argc)
                   {
                       ARCSLog::getInstance()->setLogMode((ARCSLog::LogModes)QString(argv[i+1]).toInt());
                       i++;
                   }
               }

          else
               if (current == "-d" || current == "--define")
               {
               if (i+1 < argc)
               {
                    constantList = argv[i+1];
                    i++;
               }
          }
          else
          {
               if (current == "-p" || current == "--profile")
               {
                    if ( i+1 < argc)
                    {
                         profile = argv[i+1];
                         i++;
                    }
               }
               else
               {
                    if (current == "-o" || current == "--profile-out")
                    {
                         if (i+1  < argc)
                         {
                              profileOut = argv[i+1];
                              i++;
                         }

                    }
                    else
                         xmlfile = argv[i];
               }
          }
     }
}


void parseConstants(QString arglist, ARCSApplicationComponent* app)
{
     int i;
     QStringList lst = arglist.split(":");
     for (i=0; i < lst.count(); i++)
     {
      QStringList lst_2 = lst[i].split("="); // silly windows naming !
      if (lst_2.count() == 2)
	  {
           std::cout << "Overriding constant " << qPrintable(lst_2[0])
             << " with value " << qPrintable(lst_2[1]) << std::endl;
           app->changeConstant(lst_2[0], lst_2[1]);
	  }
     }
}





int main(int argc, char* argv[])
{
     ARCSApplicationComponent* app = new ARCSApplicationComponent();
     QString xmlfile ;
     QString profileFile;
     QString profileOut;
     ARCS::ARCSAppFlag flag;
     QString constantList;

     parseOptions(argc,argv, flag, constantList, xmlfile,profileFile,profileOut);
     
     if (xmlfile.isEmpty())
     {
	  usage(argv[0]);
	  return 0;
     }

     if (flag != ARCS::ARCS_APP_NONE)
          app->setCurrentMode(flag);

     if (!app->loadFile(xmlfile)) //parseString(xmlfile))
     {
	  std::cerr << "An error occured when trying to load the application" << std::endl;
	  return 0;
     }

     std::cout << "Application " << qPrintable(xmlfile) << " loaded."  << std::endl
	       << "=================================================="  << std::endl;

     if (!profileFile.isEmpty())
     {
          ARCSXMLHandler* xmlHandler= new ARCSXMLHandler();
          if (xmlHandler->openFile(profileFile))
               if (! xmlHandler->parseProfile(app->getContext()))
                    std::cerr << "Could not read profile file." << std::endl;
          delete xmlHandler;
     }

     
     if (!constantList.isEmpty())
	  parseConstants(constantList, app);


     app->startApplication();
     app->wait();
     if (!profileOut.isEmpty())
     {
          ARCSXMLHandler* xmlHandler= new ARCSXMLHandler();
          xmlHandler->storeProfile(app->getContext());
          if (!xmlHandler->saveFile(profileOut))
               std::cerr << "Could not write profile file." << std::endl;
          delete xmlHandler;
     }
    
     return 0;
}
