/*
  name: tools/editor/graphicsportitem.cpp

  This file is part of ARCS - Augmented Reality Component System
  (version 2-current), written by Jean-Yves Didier 
  for IBISC Laboratory (http://www.ibisc.univ-evry.fr)

  Copyright (C) 2013  Universit d'Evry-Val d'Essonne

  This program is free software: you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program.  If not, see <http://www.gnu.org/licenses/>.


  Please send bugreports  with examples or suggestions to
  jean-yves.didier__at__ibisc.univ-evry.fr
*/


#include "graphicsportitem.h"
#include "graphicslinkitem.h"
#include "graphicscomponentitem.h"
#include "graphicsinvocationitem.h"
#include "invocationdialog.h"
#include "sheetview.h"
#include <arcs/arcssheet.h>
#include <QBrush>
#include <QGraphicsSceneMouseEvent>
#include <iostream>
#include <QGraphicsScene>
#include <QAction>
#include <QMenu>
#include <QRegExp>
#include <QMetaObject>
#include <QMessageBox>
#include <QStyleOptionGraphicsItem>

#include <QInputDialog>

GraphicsPortItem::GraphicsPortItem(QGraphicsItem *parent)
    : QGraphicsPathItem(parent)
{
    markedForDeletion= false;
    linkInConstruction = 0;

#if QT_VERSION >= 0x050000
    setAcceptHoverEvents(true);
#else
    setAcceptsHoverEvents(true);
#endif
    setFlag(QGraphicsItem::ItemIsSelectable);
    setFlag(QGraphicsItem::ItemSendsScenePositionChanges);
    defaultPen = pen();

    QBrush br=brush();
    br.setStyle(Qt::SolidPattern);
    br.setColor(Qt::white);

    setBrush(br);

    QPainterPath pp ;

    pp.addRect(-5,-5,10,10);
    pp.moveTo(-5,0);
    pp.lineTo(5,0);
    pp.moveTo(5,0);
    pp.lineTo(0,-3);
    pp.moveTo(5,0);
    pp.lineTo(0,3);

    setPath(pp);

    setAcceptHoverEvents(true);
    rectOperation = new QGraphicsRectItem(this);
    textOperation = new QGraphicsSimpleTextItem(rectOperation);
    rectOperation->setVisible(false);
    rectOperation->setRect(textOperation->boundingRect());
    rectOperation->setBrush(QBrush(Qt::yellow));
    //setFlag(QGraphicsItem::ItemIsMovable,false);
    updateView();
}

GraphicsPortItem::~GraphicsPortItem()
{
    setSelected(false);
    int i;
     // when a portitem is removed, it should destroy all links it is connected with.
    for(i=0; i < links.count(); i++)
    {
        if (markedForDeletion)
            links[i]->markForDeletion();
        delete links[i];
    }

    // we should also mark for deletion invocations
    if (markedForDeletion)
    {
#if QT_VERSION >= 0x050000
        QList<QGraphicsItem*> items = childItems();
#else
        QList<QGraphicsItem*> items = children();
#endif
        for (i=0; i < items.count(); i++ )
        {
            if (items[i]->type() == GraphicsInvocationItem::Type)
                dynamic_cast<GraphicsInvocationItem*>(items[i])->markForDeletion();
        }
    }

    // we should tell  our father we do not enlist anymore.
    GraphicsComponentItem* parentComponent = dynamic_cast<GraphicsComponentItem*>(parentItem());
    if (parentComponent)
        parentComponent->removePort(textOperation->text());
}


void GraphicsPortItem::updateView()
{
    textOperation->setText(operationName);
    rectOperation->setRect(textOperation->boundingRect().adjusted(-2,-2,2,2));
    QSizeF s = rectOperation->rect().size();

    if (operationType == Slot)
        rectOperation->setPos(-s.width()-15,-s.height() -10);
    else
        rectOperation->setPos(15,-s.height()-10);
}


void GraphicsPortItem::hoverEnterEvent ( QGraphicsSceneHoverEvent * /*event*/ )
{
    showName();
}

void GraphicsPortItem::hoverLeaveEvent ( QGraphicsSceneHoverEvent * /*event*/ )
{
    hideName();
}

void GraphicsPortItem::mousePressEvent(QGraphicsSceneMouseEvent *event)
{
    linkInConstruction = new GraphicsLinkItem(this);

    if (operationType== Slot)
        linkInConstruction->setDestination(this);
    else
        linkInConstruction->setSource(this);

    QGraphicsPathItem::mousePressEvent(event);
}


void GraphicsPortItem::mouseMoveEvent(QGraphicsSceneMouseEvent *event)
{
    if (linkInConstruction)
    {
        linkInConstruction->setCurrentPoint(event->pos());
    }


}


void GraphicsPortItem::mouseReleaseEvent(QGraphicsSceneMouseEvent *event)
{

    //QGraphicsItem* item = scene()->itemAt(event->scenePos());
    QList<QGraphicsItem*> items = scene()->items(event->scenePos(), Qt::IntersectsItemBoundingRect, Qt::DescendingOrder);
    QGraphicsItem* item = 0;
    for (int i=0; i < items.count(); i++)
    {
        if (items[i]->type() == GraphicsPortItem::Type)
            item = items[i];
    }


    if (item && item != this)
    {
            GraphicsPortItem* portItem = dynamic_cast<GraphicsPortItem*>(item);


            if (portItem->getType() != this->getType())
            {
                if ( portItem->getType() == Slot)
                    linkInConstruction->setDestination(portItem);
                else
                    linkInConstruction->setSource(portItem);

                // put here more test, for example compatibility between signature of each signal/slot pair
                if (!QMetaObject::checkConnectArgs(
                            qPrintable(linkInConstruction->getDestination()->getName()),
                            qPrintable(linkInConstruction->getSource()->getName()) ) )
                {
                    if ( QMessageBox::question(0,"Connect argument mismatch",
                                          "Arguments from your signal and your slot mismatch. Do you want to continue ?",
                                          QMessageBox::Yes, QMessageBox::No) == QMessageBox::No)
                    {
                        delete linkInConstruction;
                        linkInConstruction = 0;
                    }

                }

                linkInConstruction->finalize();
                links.append(linkInConstruction);
                portItem->links.append(linkInConstruction);
            }


            if (item == this)
            {
                QGraphicsPathItem::mouseReleaseEvent(event);
            }


    }

    if (!linkInConstruction->isFinalized())
    {
        delete linkInConstruction;
        linkInConstruction = 0;
    }

}


void GraphicsPortItem::contextMenuEvent(QGraphicsSceneContextMenuEvent *event)
{
     if (operationType == Signal)
          return ;
     QMenu menu ;
     /*QAction* addInvocation =*/ menu.addAction("Add invocation");


     QAction* selectedAction = menu.exec(event->screenPos());
     if (!selectedAction)
         return;

     //QString text = QInputDialog::getText(0,"Enter an invocation value","Enter an invocation value");
     /*if (text.isEmpty())
         return;*/
     // put here something in order to parse the invocation

     InvocationDialog dialog(operationName);

     if (dialog.exec() == QDialog::Accepted)
     {
         GraphicsComponentItem* component = dynamic_cast<GraphicsComponentItem*>(parentItem());

         SheetView* sheetView = dynamic_cast<SheetView*>(scene()->views()[0]);
         if (sheetView)
         {
              ARCSSheet& sheet = sheetView->getSheet();
              ARCSInit* init=0;
              switch(dialog.getInvocationType())
              {
              case GraphicsInvocationItem::Pre:
                  init = &sheet.addPreconnect(component->getComponent()->getProperty("id").toString(),
                                             operationName,dialog.getTypeName(),dialog.getValue());
                  break;
              case GraphicsInvocationItem::Post:
                  init = &sheet.addPostconnect(component->getComponent()->getProperty("id").toString(),
                                              operationName,dialog.getTypeName(),dialog.getValue());
                  break;
              case GraphicsInvocationItem::Cleanup:
                  init = &sheet.addCleanup(component->getComponent()->getProperty("id").toString(),
                                          operationName,dialog.getTypeName(),dialog.getValue());
                  break;
              }
              if (init )
                new GraphicsInvocationItem(this,dialog.getInvocationType(),*init);
         }
     }

     std::cout << "invocation added" << std::endl;
}


QVariant GraphicsPortItem::itemChange(GraphicsItemChange change, const QVariant &value)
{
    if (change == ItemChildRemovedChange)
    {
    }
    return QGraphicsItem::itemChange(change,value);

}

void GraphicsPortItem::paint(QPainter *painter, const QStyleOptionGraphicsItem *option, QWidget *widget)
{
    QStyleOptionGraphicsItem myOptions(*option);
    if (myOptions.state & (QStyle::State_Selected))
    {
        QPen myPen(Qt::darkRed);
        myPen.setCapStyle(Qt::RoundCap);
        myPen.setWidth(2);
        setPen(myPen);
        myOptions.state = myOptions.state ^ QStyle::State_Selected  ;
    }
    else
    {
        setPen(defaultPen);
    }
    QGraphicsPathItem::paint(painter,& myOptions,  widget );
}

void GraphicsPortItem::setName(QString s)
{
    operationName = s;
    updateView();
}


void GraphicsPortItem::fireUpdate()
{
    //std::cout << "Redrawing all children" << std::endl;
    for (int i=0; i< links.count(); i++)
    {
        //links[i]->setParentItem(this);
        links[i]->redraw(/*operationType == Signal*/);
    }
}

void GraphicsPortItem::removeLinkItem(GraphicsLinkItem *item)
{
     links.removeAll(item);
}
