/*
  name: tools/editor/graphicscomponentitem.cpp

  This file is part of ARCS - Augmented Reality Component System
  (version 2-current), written by Jean-Yves Didier 
  for IBISC Laboratory (http://www.ibisc.univ-evry.fr)

  Copyright (C) 2013  Universit d'Evry-Val d'Essonne

  This program is free software: you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program.  If not, see <http://www.gnu.org/licenses/>.


  Please send bugreports  with examples or suggestions to
  jean-yves.didier__at__ibisc.univ-evry.fr
*/


#include "graphicscomponentitem.h"
#include "sheetview.h"
#include "editcomponentwidget.h"
#include <QFont>
#include <QMenu>
#include <QGraphicsSceneContextMenuEvent>
#include <QMessageBox>
#include <QFontMetrics>
#include <QStringList>
#include <QStyle>
#include <QStyleOptionGraphicsItem>


#include "portdialog.h"
#include <iostream>

GraphicsComponentItem::GraphicsComponentItem(ARCSAbstractComponent *aac)
    : QGraphicsRectItem()
{
    markedForDeletion = false;
    defaultPen = pen();
    setAcceptHoverEvents(true);

    setFlag(QGraphicsItem::ItemIsMovable);
    setFlag(QGraphicsItem::ItemIsSelectable);
    setFlag(QGraphicsItem::ItemSendsScenePositionChanges);

    component = aac;

    componentLabel = new QGraphicsSimpleTextItem(aac->getProperty("id").toString()+":"+aac->getType(),this);
    setRect(componentLabel->boundingRect().adjusted(-5,-5,5,5));
}

GraphicsComponentItem::~GraphicsComponentItem()
{
    setSelected(false);
    if (markedForDeletion)
    {


        QMapIterator<QString,GraphicsPortItem*> i(signalPorts);
        while (i.hasNext())
        {
            i.next();
            i.value()->markForDeletion();
        }
        QMapIterator<QString,GraphicsPortItem*> i2(slotPorts);
        while (i2.hasNext())
        {
            i2.next();
            i2.value()->markForDeletion();
        }


        SheetView* sheetView = dynamic_cast<SheetView*>(scene()->views()[0]);
        if (sheetView)
        {
            sheetView->getSheet().removeProperty(component->getProperty("id").toString() + "-slots");
            sheetView->getSheet().removeProperty(component->getProperty("id").toString() + "-signals");
            sheetView->getSheet().removeProperty(component->getProperty("id").toString() + "-pos");
        }
    }
}


void GraphicsComponentItem::hoverEnterEvent ( QGraphicsSceneHoverEvent * /*event*/ )
{
    setPen(QPen(Qt::darkBlue));
    update();
}


void GraphicsComponentItem::hoverLeaveEvent ( QGraphicsSceneHoverEvent * /*event*/ )
{
    setPen(defaultPen);
    update();
}



void GraphicsComponentItem::removePort(QString name)
{

    slotPorts.remove(name);
    signalPorts.remove(name);

    SheetView* sheetView = dynamic_cast<SheetView*>(scene()->views()[0]);
    if (sheetView)
    {
        sheetView->getSheet().setProperty(component->getProperty("id").toString() + "-slots",
                                          QStringList(slotPorts.keys()).join(";"));
        sheetView->getSheet().setProperty(component->getProperty("id").toString() + "-signals",
                                          QStringList(signalPorts.keys()).join(";"));
    }

}

void GraphicsComponentItem::contextMenuEvent(QGraphicsSceneContextMenuEvent* event)
{
    QMenu menu;

    QAction* addSlot = menu.addAction("Add slot");
    /*QAction* addSignal  =*/ menu.addAction("Add signal");
    menu.addSeparator();

    QAction* selectedAction = menu.exec(event->screenPos());

    if (!selectedAction)
        return ;

    bool type = (selectedAction==addSlot);

    PortDialog pd(this->component,type);

    pd.exec();

    if (pd.result() == QDialog::Accepted)
    {
        if (type)
        {
            // in this case we have a slot
            if (slotPorts.contains(pd.getPort()))
            {
                QMessageBox::information(0,"Port already added",
                                         "The port "+ pd.getPort() + " has already been added to this component.",
                                         QMessageBox::Ok);
                return;
            }
            this->addSlot(pd.getPort());
        }
        else
        {
             if (signalPorts.contains(pd.getPort()))
             {
                 QMessageBox::information(0,"Port already added",
                                          "The port "+ pd.getPort() + " has already been added to this component.",
                                          QMessageBox::Ok);
                 return;
             }
             this->addSignal(pd.getPort());
        }
        // ajouter slot ou signal
    }
}


void GraphicsComponentItem::addSignal(QString s, bool updateSheet)
{
     GraphicsPortItem* item = new GraphicsPortItem(this);
     item->setType(GraphicsPortItem::Signal);
     item->setName(s);

     signalPorts[s] = item;
     item->setPos(rect().width()-5,QFontMetrics(componentLabel->font()).height()+15*(signalPorts.count()-1)+5);

     QRectF rectangle(rect());
     rectangle.setHeight(QFontMetrics(componentLabel->font()).height()
                         + 15*(  (signalPorts.count()>slotPorts.count())?signalPorts.count():slotPorts.count() )+5);
     setRect(rectangle);

     if (updateSheet)
     {
        SheetView* sheetView = dynamic_cast<SheetView*>(scene()->views()[0]);
        if (sheetView)
             sheetView->getSheet().setProperty(component->getProperty("id").toString() + "-signals",
                                               QStringList(signalPorts.keys()).join(";"));
     }

}


void GraphicsComponentItem::addSlot(QString s, bool updateSheet)
{
     GraphicsPortItem* item = new GraphicsPortItem(this);
     item->setType(GraphicsPortItem::Slot);
     item->setName(s);

     slotPorts[s] = item;
     item->setPos(-5,QFontMetrics(componentLabel->font()).height()+15*(slotPorts.count()-1)+5);

     QRectF rectangle(rect());
     rectangle.setHeight(QFontMetrics(componentLabel->font()).height()
                         + 15*( (signalPorts.count()>slotPorts.count())?signalPorts.count():slotPorts.count())+5);
     setRect(rectangle);

     if (updateSheet)
     {
        SheetView* sheetView = dynamic_cast<SheetView*>(scene()->views()[0]);
        if (sheetView)
             sheetView->getSheet().setProperty(component->getProperty("id").toString() + "-slots",
                                               QStringList(slotPorts.keys()).join(";"));
     }

}


QVariant GraphicsComponentItem::itemChange ( GraphicsItemChange change, const QVariant & value )
{
    if ( change == ItemScenePositionHasChanged  && scene() )
    {
        QMap<QString,GraphicsPortItem*>::iterator i;
        for (i = slotPorts.begin(); i != slotPorts.end(); ++i)
            (i.value())->fireUpdate();
        for (i = signalPorts.begin(); i != signalPorts.end(); ++i)
            (i.value())->fireUpdate();

        SheetView* sheetView = dynamic_cast<SheetView*>(scene()->views()[0]);
        if (sheetView)
             sheetView->getSheet().setProperty(component->getProperty("id").toString() + "-pos",
                                               QString::number(x())+","+QString::number(y()));

    }

    if (change == ItemChildRemovedChange)
    {
        // we shall recompute layout for ports !
        //! \todo  ports are always sorted in a dictionnary order whereas we add them not necessarily in a dictionnary order !
        int j=0;
        QMap<QString,GraphicsPortItem*>::iterator i;
        for (i = slotPorts.begin(); i != slotPorts.end(); ++i)
        {
            i.value()->setPos(-5,QFontMetrics(componentLabel->font()).height()+15*j+5);
            (i.value())->fireUpdate();
            j++;
        }
        j=0;
        for (i = signalPorts.begin(); i != signalPorts.end(); ++i)
        {
            i.value()->setPos(rect().width()-5,QFontMetrics(componentLabel->font()).height()+15*j+5);
            (i.value())->fireUpdate();
            j++;
        }

        //! \todo warning: lines below are a code replicate with subtle changes
        QRectF rectangle(rect());

        rectangle.setHeight(QFontMetrics(componentLabel->font()).height()
                            + 15*( (signalPorts.count()>slotPorts.count())?signalPorts.count():slotPorts.count())+5);
        setRect(rectangle);

        SheetView* sheetView = dynamic_cast<SheetView*>(scene()->views()[0]);
        if (sheetView)
            sheetView->getSheet().setProperty(component->getProperty("id").toString() + "-slots",
                                              QStringList(slotPorts.keys()).join(";"));

    }


    return QGraphicsRectItem::itemChange(change,value);
}


void GraphicsComponentItem::paint(QPainter *painter, const QStyleOptionGraphicsItem *option, QWidget *widget)
{
    QStyleOptionGraphicsItem myOptions(*option);
    if (myOptions.state & (QStyle::State_Selected))
    {
        QPen myPen(Qt::darkRed);
        myPen.setWidth(2);
        setPen(myPen);
        componentLabel->setPen(QPen(Qt::darkRed));
        myOptions.state = myOptions.state ^ QStyle::State_Selected  ;
    }
    else
    {
        setPen(defaultPen);
        componentLabel->setPen(Qt::NoPen);
    }
    QGraphicsRectItem::paint(painter,& myOptions,  widget );

}


void GraphicsComponentItem::mouseDoubleClickEvent(QGraphicsSceneMouseEvent */*event*/)
{
    // here we make an editable component.
    SheetView* sheetView = dynamic_cast<SheetView*>(scene()->views()[0]);
    if (sheetView)
        sheetView->sendWidget(new EditComponentWidget(component,sheetView));
}
