/*
  name: tools/editor/applicationview.cpp

  This file is part of ARCS - Augmented Reality Component System
  (version 2-current), written by Jean-Yves Didier 
  for IBISC Laboratory (http://www.ibisc.univ-evry.fr)

  Copyright (C) 2013  Universit d'Evry-Val d'Essonne

  This program is free software: you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program.  If not, see <http://www.gnu.org/licenses/>.


  Please send bugreports  with examples or suggestions to
  jean-yves.didier__at__ibisc.univ-evry.fr
*/


#include "applicationview.h"
#include <QVBoxLayout>
#include <QLabel>
#include <QComboBox>
#include <QMenu>
#include <QTreeWidget>
#include <QFileDialog>
#include <QMessageBox>
#include <QInputDialog>

#include <QDrag>
#include <QMimeData>


#include <arcs/arcsfactory.h>

#include "newcomponentdialog.h"
#include "editcomponentwidget.h"
#include "applicationitems.h"
#include "sheetview.h"

ApplicationView::ApplicationView(ARCSApplicationComponent *aac, QWidget *parent) :
    QDockWidget("Application view", parent)
{
    if (aac)
        application = aac;
    else
        application = new ARCSApplicationComponent();

    setProperty("appId", reinterpret_cast<quint64>(application) );

    connect(this,SIGNAL(addWidget(QWidget*)),parent,SLOT(addNewTabWidget(QWidget*)));

    setAllowedAreas(Qt::LeftDockWidgetArea | Qt::RightDockWidgetArea);
    setFloating(false);
    //application = new ARCSApplicationComponent();

    setWidget(new QWidget());
    QVBoxLayout* vl = new QVBoxLayout();
    widget()->setLayout(vl);
    vl->addWidget(new QLabel("Application mode :"));


    modeList= new QComboBox(this);
    modeList->addItem("Base",QVariant(ARCS::ARCS_APP_BASE));
    modeList->addItem("Event",QVariant(ARCS::ARCS_APP_EVENT));
    modeList->addItem("Thread",QVariant(ARCS::ARCS_APP_THREAD));
    modeList->addItem("Thread event",QVariant(ARCS::ARCS_APP_THREADEVENT));
    modeList->addItem("Gui",QVariant(ARCS::ARCS_APP_GUI));

    vl->addWidget(modeList);
    modeList->setCurrentIndex(0);

    connect(modeList,SIGNAL(currentIndexChanged(int)),this,SLOT(setApplicationMode(int)));

    buildTreeView();
    vl->addWidget(treeView);


    connect(treeView,SIGNAL(itemPressed(QTreeWidgetItem*,int)),this,SLOT(handleTreeViewClic(QTreeWidgetItem*,int)));
    connect(treeView,SIGNAL(itemDoubleClicked(QTreeWidgetItem*,int)),this,SLOT(handleTreeViewDoubleClic(QTreeWidgetItem*,int)));

    selectedCandidate = 0;
    updateApplication();
}

ApplicationView::~ApplicationView()
{
    // Here application view is closed.
    emit requestAppIdDestroy(property("appId"));
}


void ApplicationView::updateApplication()
{
    if (application->getCurrentMode() == ARCS::ARCS_APP_NONE)
        application->setCurrentMode(ARCS::ARCS_APP_EVENT);

    if (application->getProcessCount() == 0)
    {
        // il faut ajouter  l'application un process.
        //! \todo ceci ne serait valable que dans le cadre d'un dmarrage  zro.
        ARCSProcess* prc = new ARCSProcess();
        prc->setContext(application->getContext());
        application->addProcess(prc);
        processList->addChild(new ProcessItem());
    }
    else
    {
        int i,j,k;
        // we should get all components.
        QStringList components = application->getContext()->getComponentList();

        componentList->takeChildren();
        processList->takeChildren();


        for(i=0; i < components.count(); i++)
        {
            QTreeWidgetItem* cItem = new ComponentItem(components[i],application->getContext()->getComponent(components[i])->getType());
            componentList->addChild(cItem);
        }
        componentList->sortChildren(0,Qt::AscendingOrder);

        //! \todo do not forget to load constants.

        // then we should get all processes.
        k = application->getProcessCount();
        for (i=0; i< k;i++)
        {
            ARCSProcess* proc = application->getProcess(i);
            ProcessItem* procItem = new ProcessItem() ;
            processList->addChild(procItem);
            procItem->setController(proc->getControllerId());

            QStringList sheets = proc->getSheetNames();
            for (j=0; j < sheets.count(); j++)
            {
                procItem->addChild(new SheetItem(sheets[j]));
            }
        }
    }
    treeView->expandAll();
    modeList->setCurrentIndex(modeList->findData(QVariant(application->getCurrentMode())));

}

void ApplicationView::setApplicationMode(int mode)
{
    application->setCurrentMode(static_cast<ARCS::ARCSAppFlag>(modeList->itemData(mode).toInt()));
}


void ApplicationView::buildTreeView()
{
    treeView = new QTreeWidget(this);
    treeView->setHeaderLabel("application");

    QTreeWidgetItem* appItem = new QTreeWidgetItem(QStringList("application"));
    appItem->setIcon(0,QIcon(":icons/document-open-remote.png"));
    QTreeWidgetItem* ctxItem = new QTreeWidgetItem(QStringList("context"));
    appItem->addChild(ctxItem);  
    processList = new QTreeWidgetItem(QStringList("processes"));
    appItem->addChild(processList);
    processList->setIcon(0,QIcon(":icons/system-run.png"));

    componentList = new QTreeWidgetItem(QStringList("components"));
    componentList->setIcon(0,QIcon(":icons/code-block.png"));
    ctxItem->addChild(componentList);
    QTreeWidgetItem* constantList = new QTreeWidgetItem(QStringList("constants"));
    ctxItem->addChild(constantList);
    constantList->setIcon(0,QIcon(":icons/irc-join-channel.png"));

    treeView->insertTopLevelItem(0,appItem);




    // fondamentalement les actions devraient tre effectivement groupes de manire diffrente
    // par rapport au treeview.

    //QMenu* menu = new QMenu(treeView);
    QAction* actionDefault = new QAction("Set as active application",this);
    connect(actionDefault, SIGNAL(triggered()),this,SLOT(requestDefaultOwnership()));
    QAction* action = new QAction("Add library",this);
    connect(action,SIGNAL(triggered()),this,SLOT(addLibrary()));
    QAction* action2 = new QAction("Add component",this);
    connect(action2,SIGNAL(triggered()),this,SLOT(addComponent()));
    removeComponentAction = new QAction("Remove component",this);
    connect(removeComponentAction,SIGNAL(triggered()),this,SLOT(removeComponent()));
    renameComponentAction = new QAction("Rename component...",this);
    connect(renameComponentAction,SIGNAL(triggered()),this,SLOT(renameComponent()));
    editComponentAction = new QAction("Edit component",this);
    connect(editComponentAction,SIGNAL(triggered()),this,SLOT(editComponent()));
    QAction* action3 = new QAction("Add process",this);
    connect(action3,SIGNAL(triggered()),this,SLOT(addProcess()));
    attachControllerAction = new QAction("Attach controller...",this);
    connect(attachControllerAction,SIGNAL(triggered()),this,SLOT(attachController()));

    addSheetAction = new QAction("Add sheet",this);
    connect(addSheetAction,SIGNAL(triggered()),this,SLOT(addSheet()));
    editSheetAction = new QAction("Edit sheet",this);
    connect(editSheetAction, SIGNAL(triggered()),this,SLOT(editSheet()));
    removeSheetAction = new QAction("Remove sheet",this);
    connect(removeSheetAction, SIGNAL(triggered()),this,SLOT(removeSheet()));
    renameSheetAction = new QAction("Rename sheet",this);
    connect(renameSheetAction, SIGNAL(triggered()),this,SLOT(renameSheet()));

    moveProcessUpAction= new QAction("Move Process up", this);
    connect(moveProcessUpAction,SIGNAL(triggered()),this,SLOT(moveProcessUp()));
    moveProcessDownAction= new QAction("Move Process down", this);
    connect(moveProcessDownAction,SIGNAL(triggered()),this,SLOT(moveProcessDown()));

    removeProcessAction= new QAction("Delete Process", this);
    connect(removeProcessAction,SIGNAL(triggered()),this,SLOT(removeProcess()));

    sheetSeparatorAction = new QAction(this);
    sheetSeparatorAction->setSeparator(true);

    processSeparatorAction = new QAction(this);
    processSeparatorAction->setSeparator(true);

    componentSeparatorAction = new QAction(this);
    componentSeparatorAction->setSeparator(true);


    treeView->addAction(actionDefault);
    treeView->addAction(action);
    treeView->addAction(action2);
    treeView->addAction(processSeparatorAction);
    treeView->addAction(moveProcessUpAction);
    treeView->addAction(moveProcessDownAction);
    treeView->addAction(removeProcessAction);
    treeView->addAction(componentSeparatorAction);
    treeView->addAction(renameComponentAction);
    treeView->addAction(removeComponentAction);
    treeView->addAction(editComponentAction);
    treeView->addAction(attachControllerAction);
    treeView->addAction(action3);    
    treeView->addAction(addSheetAction);
    treeView->addAction(sheetSeparatorAction);
    treeView->addAction(editSheetAction);
    treeView->addAction(renameSheetAction);
    treeView->addAction(removeSheetAction);
    /*editComponentAction->setVisible(false);
    removeComponentAction->setVisible(false);
    moveProcessUpAction->setVisible(false);
    moveProcessDownAction->setVisible(false);
    removeProcessAction->setVisible(false);
    removeSheetAction->setVisible(false);*/

    treeView->setContextMenuPolicy(Qt::ActionsContextMenu);
    //items.append();
}

void ApplicationView::addLibrary()
{
    QString res = QFileDialog::getOpenFileName(this,"Get a library",QString(),QString("*.so"));
    if (res.isEmpty())
        return ;

    /*bool loaded  =*/  ARCSFactory::getInstance()->loadLibrary(res);

    //if (!loaded)

   /* QList<QTreeWidgetItem*> items = treeView->findItems("context",Qt::MatchExactly) ;
    if (items.count() == 0)
    {
        QMessageBox::critical(this,"Pas d'lment library ","");
        return ;
    }
    items.at(0)->addChild(new QTreeWidgetItem(QStringList(res)));*/
}



void ApplicationView::addComponent()
{
    NewComponentDialog  dlg(application->getContext(),this);
    if (dlg.exec())
    {
        QTreeWidgetItem* cItem = new ComponentItem(dlg.getName(),dlg.getType());
        componentList->addChild(cItem);
        componentList->sortChildren(0,Qt::AscendingOrder);

        //! \todo ne devrait on pas dlguer la cration du composant  la bote de dialogue ? -> non car cela permet de crer les composants sans passer par la bote comme par exemple lorsqu'un fichier d'application est charg.

        ARCSContext* ctx = application->getContext();
        ctx->createComponent(dlg.getName(),dlg.getType());
    }
}


void ApplicationView::handleTreeViewClic(QTreeWidgetItem *item, int /*column*/)
{
    selectedCandidate = item;
    bool status;

    status = (item->type() == ComponentItem::Type) ;
    componentSeparatorAction->setVisible(status);
    removeComponentAction->setVisible(status);
    editComponentAction->setVisible(status);
    renameComponentAction->setVisible(status);

    if (status)
    {
        // voir pour mettre un drag en place.
        QDrag *drag = new QDrag(this);
        QMimeData *mimeData = new QMimeData;

        /* a voir en fonction des donnes  transfrer */
        mimeData->setText(selectedCandidate->text(1));
        drag->setMimeData(mimeData);
        /*Qt::DropAction dropAction =*/ drag->exec();
    }

    status = (item->type() == ProcessItem::Type) ;
    processSeparatorAction->setVisible(status);
    attachControllerAction->setVisible(status);
    addSheetAction->setVisible(status);
    moveProcessUpAction->setVisible(status);
    moveProcessDownAction->setVisible(status);
    removeProcessAction->setVisible(status);

    status = (item->type() == SheetItem::Type);
    sheetSeparatorAction->setVisible(status);
    editSheetAction->setVisible(status);
    removeSheetAction->setVisible(status);
    renameSheetAction->setVisible(status);



}


void ApplicationView::handleTreeViewDoubleClic(QTreeWidgetItem* item, int /*column*/)
{
    selectedCandidate = item;

    if (item->type() == SheetItem::Type)
        editSheet();

    if (item->type() == ComponentItem::Type)
        editComponent();

}



void ApplicationView::removeComponent()
{
    if (selectedCandidate)
    {
        ComponentItem* ci = dynamic_cast<ComponentItem*> (selectedCandidate);
        if (ci)
        {
            application->removeComponent(ci->getName());
            emit requestUpdate();
            updateApplication();
        }
        delete selectedCandidate;
    }
}

void ApplicationView::renameComponent()
{
    if (selectedCandidate)
    {
        ComponentItem* ci = dynamic_cast<ComponentItem*> (selectedCandidate);
        if (ci)
        {
            QString newName;
            do
            {
                newName = QInputDialog::getText(this,"Rename Component",
                                                     "Please give a new name to the component. \n (Be aware that if you choose a name that already exists, we will ask you again to pick a new name)");
            }
            while(application->getContext()->getComponentList().contains(newName));

            if (newName.isEmpty())
                return ;



            if (application->getContext()->renameComponent(ci->getName(),newName))
            {
                emit requestUpdate();
                updateApplication();
            }
        }
    }
}


void ApplicationView::editComponent()
{
    if (selectedCandidate)
    {
        ComponentItem* ci = dynamic_cast<ComponentItem*> (selectedCandidate);
        if (ci)
        {
            ARCSContext* ctx = application->getContext();
            ARCSAbstractComponent* aac = ctx->getComponent(ci->getName());
            EditComponentWidget* ecw = new EditComponentWidget(aac,this);
            ecw->setProperty("appId",this->property("appId"));
            emit addWidget(ecw);
        }
    }
}



void ApplicationView::addProcess()
{
    processList->addChild(new ProcessItem());
}

void ApplicationView::removeProcess()
{
    if (selectedCandidate)
    {
        ProcessItem* pi = dynamic_cast<ProcessItem*>(selectedCandidate);
        if (pi)
        {
            int idx = processList->indexOfChild(selectedCandidate);
            application->removeProcess(idx);
            processList->removeChild(selectedCandidate);
        }
    }
}

void ApplicationView::closeEvent(QCloseEvent *event)
{
    QDockWidget::closeEvent(event);
    deleteLater();
}


void ApplicationView::attachController()
{
    if (selectedCandidate)
    {
        ProcessItem* pi = dynamic_cast<ProcessItem*>(selectedCandidate);
        if (pi)
        {
            ARCSContext* ctx = application->getContext();
            QStringList components = ctx->getComponentList();
            QStringList controllers;
            for (int i=0; i < components.count(); i++)
            {
                ARCSAbstractComponent* aac = ctx->getComponent(components[i]);
                if ( aac->getType() == "StateMachine" )
                {
                    controllers << aac->getProperty("id").toString();
                }
            }
            if (!controllers.count())
            {
                QMessageBox::information(this,"Controllers not found","Controllers not found");
                return ;
            }
            bool ok;
            QString res = QInputDialog::getItem(this,"Attaching a controller","Please pick one controller in the list below",controllers,0,false,&ok);
            if (ok && !res.isEmpty())
            {
                pi->setController(res);
                // here we are missing informations about the process
                int idx=0;
                while ( processList->child(idx) != pi ) { idx++; }
                application->getProcess(idx)->setController(res);
            }
        }
    }
}


void ApplicationView::editSheet()
{
    if (selectedCandidate)
    {
        SheetItem* ci = dynamic_cast<SheetItem*> (selectedCandidate);
        if (ci)
        {
            //ARCSContext* ctx = application->getContext();
            //ARCSAbstractComponent* aac = ctx->getComponent(ci->getName());
            ProcessItem* pi = dynamic_cast<ProcessItem*> (ci->parent());
            // first, look for process id.
            int idx=0;
            // might go out of bounds if not correctly checked.
            while ( processList->child(idx) != pi ) { idx++; }
            ARCSProcess* process = application->getProcess(idx);

            ARCSSheet& st = process->getSheet(ci->getName());
            SheetView * sv = new SheetView(st, this);
            sv->setWindowTitle("Sheet: " + st.getName());
            sv->setProperty("appId",this->property("appId"));
            connect(sv,SIGNAL(requestApplicationUpdate()), this, SLOT(updateApplication()));
            connect(this,SIGNAL(requestUpdate()),sv,SLOT(updateSheet()));
                emit addWidget(sv);

        }
    }
}

void ApplicationView::renameSheet()
{
    if (selectedCandidate)
    {
        SheetItem* si = dynamic_cast<SheetItem*>(selectedCandidate);
        if (si)
        {
            ProcessItem* pi = dynamic_cast<ProcessItem*> (si->parent());
            // first, look for process id.
            int idx=processList->indexOfChild(pi);
            ARCSProcess* process = application->getProcess(idx);
            QStringList sheetList = process->getSheetNames();

            QString newSheetName;
            do
            {
                newSheetName = QInputDialog::getText(this,"Rename Sheet",
                                                     "Please give a new name to the sheet. \n (Be aware that if you choose a name that already exists, we will ask you again to pick a new name)");
            }
            while(sheetList.contains(newSheetName));
            if (newSheetName.isEmpty())
                return ;

            // we should ask for deletion of widgets with this sheet name.
            emit requestWidgetDestroy(property("appId"),"Sheet: " + si->getName());
            //process
            //process->removeSheet(si->getName());
            QString oldSheetName = si->getName();
            process->renameSheet(oldSheetName, newSheetName);
            si->setName(newSheetName);

            //! < \todo we have to run a consistency check with the controller of the process.

            // first we should obtain the controller.
            // then we have to look if this controller is shared: it may be a source of troubles.
            // if it is the case, then we will warn the user.
            // in the other case, we just should rename sheet in transitions.
            QString controllerName = pi->getController();

            if (controllerName.isEmpty())
                return ;

            int controllerShared = 0;
            for (int i=0; i< application->getProcessCount(); i++)
            {
                if (application->getProcess(i)->getControllerId() == controllerName)
                    controllerShared++;
            }

            if (controllerShared > 1)
            {
                if ( QMessageBox::question(this,"Shared controller",
                                      "The sheet you renamed is referenced in the process controller, which is shared.\n" \
                                      "Would you like to apply changes in the controller anyway ?",
                                      QMessageBox::Yes,QMessageBox::No) == QMessageBox::No)
                    return;
            }
            ARCSControllerComponent* acc = process->getController();
            acc->getStateMachine()->renameSheet(oldSheetName,newSheetName );
        }
    }

}

void ApplicationView::removeSheet()
{
    if (selectedCandidate)
    {
        SheetItem* si = dynamic_cast<SheetItem*>(selectedCandidate);
        if (si)
        {
            ProcessItem* pi = dynamic_cast<ProcessItem*> (si->parent());
            // first, look for process id.
            int idx=pi->indexOfChild(si);
            ARCSProcess* process = application->getProcess(idx);

            // we should ask for deletion of widgets with this sheet name.
            emit requestWidgetDestroy(property("appId"),"Sheet: " + si->getName());
            process->removeSheet(si->getName());
            pi->removeChild(si);
        }
    }
}

void ApplicationView::moveProcessUp()
{
    if(selectedCandidate)
    {
        ProcessItem* pi = dynamic_cast<ProcessItem*>(selectedCandidate);
        if (pi)
        {
            int idx=processList->indexOfChild(selectedCandidate);

            if (idx > 0)
            {
                QTreeWidgetItem* item = processList->takeChild(idx);
                processList->insertChild(idx-1,item);
                application->swapProcesses(idx,idx-1);
            }
        }
    }
}



void ApplicationView::moveProcessDown()
{
    if(selectedCandidate)
    {
        ProcessItem* pi = dynamic_cast<ProcessItem*>(selectedCandidate);
        if (pi)
        {
            int idx=processList->indexOfChild(selectedCandidate);

            if (idx < processList->childCount()-1 )
            {
                QTreeWidgetItem* item = processList->takeChild(idx);
                processList->insertChild(idx+1,item);
                application->swapProcesses(idx,idx+1);
            }
        }
    }
}

void ApplicationView::addSheet()
{
    if (selectedCandidate)
    {
        ProcessItem* pi = dynamic_cast<ProcessItem*>(selectedCandidate);
        if (pi)
        {
            // first, look for process id.
            int idx=0;
            // might go out of bounds if not correctly checked.
            while ( processList->child(idx) != selectedCandidate ) { idx++; }
            ARCSProcess* process = application->getProcess(idx);


            bool ok;
            QString sheetName;
            do {
                sheetName = QInputDialog::getText(this,"Adding a new sheet", "Please give a sheet name", QLineEdit::Normal, QString(), &ok );
                if (!ok)
                    return;

                if (process->getSheetNames().contains(sheetName))
                {
                    ok = false;
                    QMessageBox::critical(this,"Sheet name error","This sheet name already exists for this component",QMessageBox::Ok,0);
                }

            } while (!ok);
            std::cout << "context pointer" << application->getContext() << std::endl;
            ARCSSheet sheet(application->getContext());
            sheet.setName(sheetName);

            process->addSheet(sheetName,sheet);

            pi->addChild(new SheetItem(sheetName));

        }

    }


}
