/*
  name: lib/arcsprocess.cpp

  This file is part of ARCS - Augmented Reality Component System
  (version 2-current), written by Jean-Yves Didier 
  for IBISC Laboratory (http://www.ibisc.univ-evry.fr)

  Copyright (C) 2013  Universit d'Evry-Val d'Essonne

  This program is free software: you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program.  If not, see <http://www.gnu.org/licenses/>.


  Please send bugreports  with examples or suggestions to
  jean-yves.didier__at__ibisc.univ-evry.fr
*/


#include <arcs/arcsprocess.h>
#include <arcs/arcsappmode.h>
#include <arcs/arcscontrollercomponent.h>

#include <QEvent>
#include <QMutex>

ARCSProcess::ARCSProcess() : QObject()
{
     controller = 0;
     currentSheetId = QString::null;
     applicationMode = 0;
     processFinished = true;
     mutex.lock();
     firstTime = true;
     mutex.unlock();
     currentMode = ARCS::ARCS_APP_NONE;
}

ARCSProcess::~ARCSProcess()
{
}

void ARCSProcess::setReferenceApplicationMode(ARCS::ARCSAppFlag flag)
{
     if (flag == ARCS::ARCS_APP_EVENT || flag == ARCS::ARCS_APP_GUI || flag == ARCS::ARCS_APP_THREADEVENT)
          currentMode = ARCS::ARCS_APP_THREADEVENT ;
     else
          currentMode = ARCS::ARCS_APP_THREAD;
     setProcessMode();
}


bool ARCSProcess::event(QEvent* event)
{
     if (event->type() == QEvent::ApplicationActivate || event->type() == ARCSAppEvent::eventType)
     {
          if (!processFinished)
               setupNextSheet();
          return true;
     }

     return QObject::event(event);
}


void ARCSProcess::changeSheet(QString s)
{
     if (!sheets.contains(s))
     {
          std::cerr << "[Process] Sheet " << qPrintable(s) << " not found." << std::endl;
          return ;
     }

     if (!currentSheetId.isEmpty())
     {
          ARCSSheet& current = sheets[currentSheetId];
          current.deactivate();
     }


     if (controller->getStateMachine()->hasFinished())
     {
          processFinished = true;
          emit finished();
          return;
     }

     currentSheetId = s;

     mutex.lock();
     if (firstTime)
     {
          firstTime = false;
          if (currentMode == ARCS::ARCS_APP_THREAD || currentMode == ARCS::ARCS_APP_THREADEVENT)
               cond.wakeAll();
     }
     mutex.unlock();

     applicationMode->tokenHook();
}


void ARCSProcess::finish()
{
     mutex.lock();
     firstTime = true;
     mutex.unlock();
}


bool ARCSProcess::setController(QString s)
{
    // may be this is not necessary
    /*if (s.isEmpty())
    {
        controllerId = s;
        controller = 0;
        return;
    }*/

     ARCSAbstractComponent* aac = context->getComponent(s);
     if (aac == 0)
          return false;

     ARCSControllerComponent* ctrl = dynamic_cast<ARCSControllerComponent*>(aac);

     if (ctrl != 0)
     {
          controller = ctrl;
          return true;
     }

     return false;
}

void ARCSProcess::setProcessMode()
{
     if (applicationMode != 0)
          delete applicationMode;

     switch(currentMode)
     {
     case ARCS::ARCS_APP_BASE:
       applicationMode = new ARCSAppBase(this);
       break;
     case ARCS::ARCS_APP_THREAD:
       applicationMode = new ARCSAppThread(this);
       break;
     case ARCS::ARCS_APP_THREADEVENT:
       applicationMode = new ARCSAppThreadEvent(this);
       break;
     case ARCS::ARCS_APP_GUI:
       applicationMode = new ARCSAppGUI(this);
       break;
     default:
       applicationMode = new ARCSAppEvent(this);
     };

}




bool ARCSProcess::startProcess()
{
     processFinished = false;

     if (controller == 0)
     {
          std::cerr << "[Core] Process has not any controller" << std::endl;
       return false;
     }

     QObject::connect(controller->getStateMachine(), SIGNAL(sendSheet(QString)), this, SLOT(changeSheet(QString)),Qt::DirectConnection);
     QObject::connect(controller->getStateMachine(), SIGNAL(finished()), this, SLOT(finish()),Qt::DirectConnection);
     applicationMode->startHook();

     return true;
}



bool ARCSProcess::stopProcess()
{
     mutex.lock();
     firstTime = true;
     mutex.unlock();
     return true;
}


void ARCSProcess::waitForFirstSheet()
{
     mutex.lock();
     if (firstTime)
          cond.wait(&mutex);
     mutex.unlock();
}

void ARCSProcess::addSheet(QString s, ARCSSheet sh)
{
     sh.setContext(context);
     sheets.insert(s, sh);
}


void ARCSProcess::setupNextSheet()
{
     if (currentSheetId.isEmpty())
          return ;

     if (!sheets.contains(currentSheetId))
     {
          std::cerr << "[Process] unknown sheet: " << qPrintable(currentSheetId) << std::endl;
          return ;
     }
     ARCSSheet& sheet = sheets[currentSheetId];
     if (!sheet.isActivated())
          sheet.activate();
}


void ARCSProcess::wait()
{
     if (currentMode != ARCS::ARCS_APP_THREAD && currentMode != ARCS::ARCS_APP_THREADEVENT)
          return ;

     if (applicationMode == NULL)
          return ;

     QThread* threadToWaitFor = dynamic_cast<QThread*>(applicationMode);

     if (threadToWaitFor == NULL)
          return ;

     threadToWaitFor->wait();
}

void ARCSProcess::renameSheet(QString oldName, QString newName)
{
    if (!sheets.contains(oldName))
        return;
    if ( sheets.contains(newName))
        return;

    ARCSSheet sheet = sheets.take(oldName);
    sheet.setName(newName);
    addSheet(newName,sheet);
}

void ARCSProcess::removeSheet(QString s)
{
    if (sheets.contains(s))
        sheets.remove(s);
}
