/*
  name: lib/arcslog.cpp

  This file is part of ARCS - Augmented Reality Component System
  (version 2-current), written by Jean-Yves Didier 
  for IBISC Laboratory (http://www.ibisc.univ-evry.fr)

  Copyright (C) 2013  Universit d'Evry-Val d'Essonne

  This program is free software: you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program.  If not, see <http://www.gnu.org/licenses/>.


  Please send bugreports  with examples or suggestions to
  jean-yves.didier__at__ibisc.univ-evry.fr
*/


#include <arcs/arcslog.h>
#include <iostream>

ARCSLogEvent::ARCSLogEvent(QString src, Level lvl, QString msg)
{
     source = src;
     level = lvl;
     message = msg;
}


QDataStream &operator<< (QDataStream &out, const ARCSLogEvent & event)
{
    out << event.getSource() << event.getLevel() << event.getMessage() << "\n" ;
    return out;
}

QDataStream &operator>> (QDataStream &in, ARCSLogEvent & event)
{
     QString src;
     QString msg;
     int lvl;
     in >> src >> lvl >> msg;
     event=ARCSLogEvent(src,(ARCSLogEvent::Level)lvl,msg);
     return in;
}

/*********************************************************************************/
// class ARCSColor Log
/*********************************************************************************/
void ARCSColorLog::log(std::ostream &) const
{
    // empty implementation by default.
}
void ARCSColorLog::log(QTextStream &) const
{
    // empty implementation by default.
}



#ifdef WIN32
WORD ARCSWindowsColorLog::getNormalAttribute()
{
     CONSOLE_SCREEN_BUFFER_INFO csbi;
     GetConsoleScreenBufferInfo (GetStdHandle(STD_OUTPUT_HANDLE),&csbi);
     return csbi.wAttributes;
}

WORD ARCSWindowsColorLog::normalAttribute= ARCSWindowsColorLog::getNormalAttribute();
WORD ARCSWindowsColorLog::attributes[] = { normalAttribute ,
                                    FOREGROUND_GREEN|FOREGROUND_INTENSITY,
                                    FOREGROUND_BLUE|FOREGROUND_INTENSITY,
                                    FOREGROUND_GREEN|FOREGROUND_RED|FOREGROUND_INTENSITY,
                                    FOREGROUND_RED|FOREGROUND_INTENSITY };

void ARCSWindowsColorLog::log(std::ostream& os) const
{
    HANDLE hd;
    if (os == std::cout)
         hd = GetStdHandle(STD_OUTPUT_HANDLE);
    else
         hd = GetStdHandle(STD_ERROR_HANDLE);
    SetConsoleTextAttribute(hd,attributes[getColor()]);
}

#define ARCSConsoleColorLog ARCSWindowsColorLog

#else
const char* ARCSUnixColorLog::normalAttribute="\033[0m";
const char* ARCSUnixColorLog::attributes[] = { normalAttribute, "\033[32;1m", "\033[34;1m", "\033[33;1m", "\033[31;1m" };

void ARCSUnixColorLog::log(std::ostream& os) const {
    os << attributes[getColor()];
}

#define ARCSConsoleColorLog ARCSUnixColorLog

#endif

const char* ARCSHTMLColorLog::normalAttribute="</font>";
const char* ARCSHTMLColorLog::attributes[] = { normalAttribute, "<font color=\"green\">", "<font color=\"blue\">", "<font color=\"orange\">",
                                               "<font color=\"red\">"};

void ARCSHTMLColorLog::log(std::ostream& os) const {
    os << attributes[getColor()];
}

void ARCSHTMLColorLog::log(QTextStream& ts) const {
    ts << attributes[getColor()];
}

std::ostream& operator<<(std::ostream& os, const ARCSColorLog& cl)
{
    // small hack here because of internal ostream changes
    if (os.rdbuf() != std::cout.rdbuf() && os.rdbuf() != std::cerr.rdbuf())
      return os;

    cl.log(os);
    return os;
}

QTextStream& operator<<(QTextStream& ts, const ARCSColorLog& cl)
{
     cl.log(ts);
     return ts;
}



/*********************************************************************************/

ARCSLog* ARCSLog::instance = 0; //new ARCSLog();

const char* ARCSLog::interp[] = { "NOR", "INF", "WRN" , "ERR" , "CRT" };

ARCSLog::ARCSLog()
{
     display = ARCSLogEvent::INFORMATION ;
     logMode = CONSOLE;
     levels[0] = 0;
     refreshMode();
     textStream = 0;

}

void ARCSLog::refreshMode()
{
    if (levels[0])
        for (int i=0; i < 5; i++)
            delete levels[i];

    if (logMode == CONSOLE)
    {
        levels[0] = new ARCSConsoleColorLog(ARCSColorLog::NORMAL);
        levels[1] = new ARCSConsoleColorLog(ARCSColorLog::GREEN);
        levels[2] = new ARCSConsoleColorLog(ARCSColorLog::BLUE);
        levels[3] = new ARCSConsoleColorLog(ARCSColorLog::YELLOW);
        levels[4] = new ARCSConsoleColorLog(ARCSColorLog::RED);
    }
    if (logMode == HTML)
    {
        levels[0] = new ARCSHTMLColorLog(ARCSColorLog::NORMAL);
        levels[1] = new ARCSHTMLColorLog(ARCSColorLog::GREEN);
        levels[2] = new ARCSHTMLColorLog(ARCSColorLog::BLUE);
        levels[3] = new ARCSHTMLColorLog(ARCSColorLog::YELLOW);
        levels[4] = new ARCSHTMLColorLog(ARCSColorLog::RED);
    }
    if (logMode == NONE)
    {
        levels[0] = new ARCSColorLog(ARCSColorLog::NORMAL);
        levels[1] = new ARCSColorLog(ARCSColorLog::GREEN);
        levels[2] = new ARCSColorLog(ARCSColorLog::BLUE);
        levels[3] = new ARCSColorLog(ARCSColorLog::YELLOW);
        levels[4] = new ARCSColorLog(ARCSColorLog::RED);
    }

}

ARCSLog* ARCSLog::getInstance()
{
     if (!instance)
          instance=new ARCSLog();

     return instance;
}


void ARCSLog::log(ARCSLogEvent evt)
{
     logEvents.append(evt);
     if (evt.getLevel() >= display)
     {
         if (textStream)
         {
             QTextStream & t = *textStream ;
             t << *levels[evt.getLevel()]
                        << "[" << interp[evt.getLevel()] << "!" << evt.getSource() <<  "] "
                        << *levels[0]
                        << evt.getMessage() << endl;
             if (logMode == HTML)
                 t << "<br/>";
         }
         else
         {

          if (evt.getLevel() >= ARCSLogEvent::WARNING)
               std::cerr << *levels[evt.getLevel()]
                         << "[" << interp[evt.getLevel()] << "!" << qPrintable(evt.getSource()) <<  "] "
                         << *levels[0]
                         << qPrintable(evt.getMessage()) << std::endl;
          else
               std::cout << *levels[evt.getLevel()]
                         << "[" << interp[evt.getLevel()] << "!" << qPrintable(evt.getSource()) << "] "
                         << *levels[0]
                         << qPrintable(evt.getMessage()) << std::endl;
         }
     }


}
