/*
  name: include/arcs/arcsxmlhandler.h

  This file is part of ARCS - Augmented Reality Component System
  (version 2-current), written by Jean-Yves Didier 
  for IBISC Laboratory (http://www.ibisc.univ-evry.fr)

  Copyright (C) 2013  Universit d'Evry-Val d'Essonne

  This program is free software: you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program.  If not, see <http://www.gnu.org/licenses/>.


  Please send bugreports  with examples or suggestions to
  jean-yves.didier__at__ibisc.univ-evry.fr
*/


#ifndef __ARCSXMLHANDLER_H__
#define __ARCSXMLHANDLER_H__

#include <QDomDocument>

#include <arcs/arcssheet.h>

#include <QFileInfo>
#include <QList>

class ARCSSheet;
class ARCSContext;
class ARCSAbstractComponent;
class ARCSCompositeComponent;
class ARCSApplicationComponent;
class ARCSStateMachine;
class ARCSFactory;
class ARCSProcess;


//! \todo Add a path stack somewhere in this class.
//! \todo Solve some troubles around strings describing components.

/*! \brief This is an XML Handler for all ARCS native XML formats.
 * 
 * The ARCSXMLHandler parses and saves all basic ARCS XML structures.
 *
 * Amongst them there are contexts, statemachines, applications and composite components.
 *
 * Here is the markup structure for contexts. A context stores the list of libraries to load,
 * a list of components to instanciate and a list of constants.
 * \dot 
 * digraph g {
 * graph [rankdir = "TB",fontname="Helvetica"];
 * edge [sametail = "true"];
 * node [shape = record, fillcolor="grey",style ="filled",fontname="sans serif"];
 * constants ;
 * node [shape=record, fillcolor="gold1", style="bold,filled", peripheries=2]
 * context ;
 * libraries ;
 * components ;
 * library [label="{library | {path}}"];
 * component [label="{component | {id |type}}"];
 * constant [label="{constant | {id|type}}"];
 * textualValue [fillcolor="white"];
 *
 * context -> libraries;
 * libraries -> library;
 * context -> components;
 * components -> component ;
 * component -> textualValue ;
 * context -> constants ;
 * constants -> constant ;
 * constant -> textualValue;
 * }
 * \enddot
 *
 * 
 * \dot
 * digraph g {
 * graph [rankdir = "TB"];
 * edge [sametail = "true"];
 * node [shape=record, fillcolor="gold1", style="bold,filled", peripheries=2,fontname="sans serif"]
 * statemachine ;
 * transitions;
 * transition [label="{transition | {source | token| destination}}"] ;
 * node [shape = record, fillcolor="grey",style ="filled"];
 * first [ label="{first | name }"];
 * last  [ label="{last|name}" ];
 * 
 * statemachine -> transitions ;
 * transitions -> transition ;
 * statemachine -> first ;

 * statemachine -> last;
 * }
 * \enddot
 *
 * \dot 
 * digraph g {
 * graph [rankdir = "TB"];
 * edge [sametail = "true"];
 * node [shape=record, fillcolor="gold1", style="bold,filled", peripheries=2,fontname="sans serif"]
 * sheet [label="{sheet|{id}}"] ;
 * connections;
 * invoke [label="{invoke|{destination|slot|type}}"];
 * link [label="{link|{source|signal|destination|slot|queued}}"];
 * node [shape = record, fillcolor="grey",style ="filled"];
 * preconnections ;
 * postconnections ;
 * connections ;
 * cleanups ;
 * textualValue [fillcolor="white"];
 * 
 * sheet -> preconnections ;
 * sheet -> connections ;
 * sheet -> postconnections ;
 * sheet -> cleanups ;
 * preconnections -> invoke ;
 * postconnections -> invoke ;
 * connections -> link ;
 * cleanups -> invoke ;
 * invoke -> textualValue ;
 * }
 * \enddot 
 *
 * \dot
 * digraph g {
 * graph [rankdir = "TB"];
 * edge [sametail = "true"];
 * node [shape=record, fillcolor="gold1", style="bold,filled", peripheries=2,fontname="sans serif"]
 * application [label="{application|{mode}}"];
 * context ;
 * processes ;
 * process [label="{process|{controller}}"];
 * sheet [label="{sheet|{id}}"] ;
 * 
 * application -> context ;
 * application -> processes;
 * processes -> process;
 * process -> sheet;
 * }
 * \enddot
 *
 * \dot 
 * digraph g {
 * graph [rankdir = "TB"];
 * edge [sametail = "true"];
 * node [shape=record, fillcolor="gold1", style="bold,filled", peripheries=2,fontname="sans serif"]
 * composite ;
 * context ;
 * sheet ;
 * interface ;
 * slots ;
 * signals ;
 * method [label="{method|{alias|component|method}}"];
 * 
 * composite -> context ;
 * composite -> sheet ;
 * composite -> interface ;
 * interface -> slots ;
 * interface -> signals ;
 * slots -> method ;
 * signals -> method;
 * }
 * \enddot
 * \author Jean-Yves Didier
 * \date January, the 22nd, 2009
 * \ingroup lib
 */
class DLL_EXPORT ARCSXMLHandler
{
public:
     //! Constructor
     ARCSXMLHandler();
     ~ARCSXMLHandler();
     
     //! This enumeration gives the list of ARCS XML structures
     enum XMLType 
     { XML_BLANK, //!< XML type not determined yet.
       XML_STATEMACHINE, //!< XML representation of a statemachine.

       XML_COMPOSITE,  //!< XML representation of a composite component.
       XML_APPLICATION,  //!< XML representation of an application.
       XML_INTERFACE, //!< XML representation of an interface
       XML_PROFILE //!< XML representation of a profile
     };



     //! Opens a file given its path. The file should describe either an application or a composite component.
     /*! \param s the file path
      * \return <tt>true</tt> if succeeded, <tt>false</tt> otherwise
      */
     bool openFile(QString s);
     //! Saves a file given its path. The file will describe either an application or a composite component.
     /*! \param s the file path
      * \return <tt>true</tt> if succeeded, <tt>false</tt> otherwise
      */
     bool saveFile(QString s);

     //! Gives the content to parse.
     /*! This is especially interesting for ...
      * \param s a string representing an xml stream to parse.
      * \return <tt>true</tt> if succeede, <tt>false</tt> otherwise
      */
     bool setContents(QString s); 
     QString getContents(); 


     void setType(XMLType t) { type = t; }
     XMLType getType(); // { return type; }


     //! Parses the xml description of a process
     /*!
      * \param ap a pointer to a process structure
      * \param processNode the XML node describing a process
      * \return <tt>true</tt> if successful, <tt>false</tt> otherwise.
      */
     bool parseProcess(ARCSProcess* ap, QDomElement processNode);

     //! Parses the xml descriptions of a sheet
     /*! 
      * \param as a pointer to a sheet structure ;
      * \param father the element containing the xml description of a sheet ;
      * \return <tt>true</tt> if sucessful, <tt>false</tt> otherwise.
      */
     bool parseSheet(ARCSSheet* as, QDomElement father); // pr ?

     //! Parses a context
     bool parseContext(ARCSContext*); // pr ?
     QStringList parseLibraries(QDomElement libraries,bool* ok=0);  // pr ?

     //! \todo determine how to use interface...
     bool parseInterface(ARCSAbstractComponent* aac,
                         QDomNode node = QDomNode());


     bool parseStateMachine(ARCSStateMachine*);
     bool parseApplication(ARCSApplicationComponent*);
     bool parseCompositeComponent(ARCSCompositeComponent*);

     bool parseProfile(ARCSContext*);

     //! Stores the xml description of a sheet
     /*! \param as a pointer to a sheet structure
      * \param father the element to which the sheet xml description will be attached.
      */
     void storeSheet(ARCSSheet* as, QDomElement father,
                     QString id=QString::null);
     void storeContext(ARCSContext*); // pr ?
     void storeStateMachine(ARCSStateMachine*);

     void storeProfile(ARCSContext*);

     void storeProcess(ARCSProcess*,QDomElement father); //{}
     void storeApplication(ARCSApplicationComponent*);
     void storeCompositeComponent(ARCSCompositeComponent*);

     //! \todo to implement, see how interfaces could be inserted as components
     void storeInterface(ARCSAbstractComponent* aac,
                         QDomElement father= QDomElement());

private:
     bool storeInvocations(ARCSSheet* sheet, ARCSSheet::GetInvocationsMethod,
                           QDomElement root);
     void storeSheetProperties(ARCSSheet* sheet,QDomElement root);

     void parseContextElement(QDomElement elt, ARCSContext* ctx,
                                     QString tagName,
                                     ARCSContext::AddPoolMethod);


     void parseSheetProperties(QDomElement elt,ARCSSheet* sheet);
     void parseInvocation(QDomElement elt, ARCSSheet* sheet,
                          ARCSSheet::AddInvocationMethod method);

     void parseMethodList(QDomElement elt,QStringList & aliases,
                                 QStringList & components,
                                 QStringList & methods);

     static QString nodeInternalsToText(QDomNode node);
     bool parseComponents(QDomElement elt, ARCSContext* ctx);

     void parseConstants(QDomElement elt, ARCSContext* ctx) {
	  parseContextElement(elt, ctx, "constant", &ARCSContext::addConstant);
     }

     // will accompany saveFile()
     void computeRelativePaths(QString basePath);
     QString solveRelativePath(QString path);

     void reset();

     //QString contents;
     XMLType type;
     QDomDocument doc;
     ARCSFactory* factory;
     QStringList libraryStrings;

     // The attributes below are used in order to properly store paths.
     QList<QDomAttr> fileAttributes; // attribute store, see saveFile()
     QString fileName; // file in which the document is stored for loadFile()


     QString logSource(int line) {
         return "XML:" + fileName + ":" + QString::number(line);
     }
     QString logSource() {
         return "XML:" + fileName;
     }

};

#endif //__ARCSXMLHANDLER_H__
