/*
  name: include/arcs/arcsqdynamicobject.h

  This file is part of ARCS - Augmented Reality Component System
  (version 2-current), written by Jean-Yves Didier 
  for IBISC Laboratory (http://www.ibisc.univ-evry.fr)

  Copyright (C) 2013  Universit d'Evry-Val d'Essonne

  This program is free software: you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program.  If not, see <http://www.gnu.org/licenses/>.


  Please send bugreports  with examples or suggestions to
  jean-yves.didier__at__ibisc.univ-evry.fr
*/


#ifndef __ARCSQDYNAMICOBJECT_H__
#define __ARCSQDYNAMICOBJECT_H__

#include <QStringList>


//! Interface to extend QObject functionnalities in order to make objects with dynamic signals or slots.
/*! Usual QObjects do not have the ability to manage dynamic signals and slots (that is to say signals or slots
 *  dynamically added to QObjects). This interface is inheriting QObjects in order to ease the implementation
 * of such QObjects so that customized QObjects can be used in the ARCS framework.
 *
 * Each class should inherit this interface. Declaration should then look like this :
 * \code
 #include <arcs/arcsqdynamicobject.h>
 class MyClass : public MyBase, public ARCSQDynamicObject
 {
 public:

     virtual int qt_metacall(QMetaObject::Call call, int id, void ** arguments);
     virtual QStringList getSignalList() ;
     virtual QStringList getSlotList() ;
     virtual int prepareSlotConnect(QString sigName, QString sltName, QString objectName = QString::null, QString actualSignal = QString::null, bool simulate = false) ;
     virtual int prepareSignalConnect(QString sigName, QString sltName, QString objectName = QString::null, QString actualSlot = QString::null, bool simulate = false);
     virtual int cleanSlotConnect(QString sigName, QString sltName, QString objectName, QString actualSignal);
     virtual int cleanSignalConnect(QString sigName, QString sltName, QString objectName, QString actualSlot);
 };
 \endcode
 * Where <tt>MyBase</tt> should inherit from <tt>QObject</tt>.
 *
 * \author Jean-Yves Didier
 * \date September, the 23rd, 2009
 * \ingroup lib
 */
class ARCSQDynamicObject //: public QObject
{
public:

     //! Method performing the actual callback task.
     /*! This method is at the core of a QObject and is usually automatically created by moc
      * Here the user has to reimplement it.
      * \param call should usually take the value QMetaObject::InvokeMetaMethod
      * \param id is the id of a slot or a signal given by indexOfSlot() or indexOfSignal()
      * \param arguments array of values (the first pointer is usually for the return value.
      * \return -1 when the call has been performed.
      */
     virtual int qt_metacall(QMetaObject::Call call, int id, void ** arguments)= 0;  //{ return QObject::qt_metacall(call,id,arguments);}

     /*! Returns the list of signals implemented inside this component. */
     virtual QStringList getSignalList() = 0;

     /*! Returns the list of slots implemented inside this component. */
     virtual QStringList getSlotList() = 0;

     //! Prepares a connection with a slot which is belonging to this object.
     /*! This method has two purposes. First it should check wether the arguments of the given signals and slots
       * are compatible. Second, it should set up what is needed in order to perform a clean connection and return
       * the slot id.
       * \param sigName name and signature of the signal which should trigger the slot.
       * \param sltName name and signature of the slot which is triggered
       * \param objectName name of the object emitting the signal
       * \param simulate if <tt>true<tt> then the actual preparation is not performed,
       * if <tt>false</tt> then everything is performed.
       * \return the id of the slot, -1 if it failed.
       */
     virtual int prepareSlotConnect( QString sigName, QString sltName, QString objectName, QString actualSignal, bool simulate = false ) = 0;

     //! Prepares a connection with a slot which does not belong to this object
     /*! This method has two purposes. First it should check wether the arguments of the given signals and slots
       * are compatible. Second, it should set up what is needed in order to perform a clean connection and return
       * the signal id.
       * \param sigName name and signature of the signal which belong to this object
       * \param sltName name and signature of the slot which is triggered by the signal
       * \param objectName name of the object which owns the slot.
       * \param simulate if <tt>true<tt> then the actual preparation is not performed,
       * if <tt>false</tt> then everything is performed.
       * \return the id of the signal, -1 if it failed.
       */
     virtual int prepareSignalConnect( QString sigName, QString sltName , QString objectName, QString actualSlot, bool simulate = false) = 0;

     /*! If a lot of things were required to prepare a slot connection, then the disconnection should also be implemented ! */
     virtual int cleanSlotConnect(QString sigName, QString sltName, QString objectName, QString actualSignal) = 0 ;

     /*! If a lot of things were required to prepare a signal connection, then the disconnection should also be implemented ! */
     virtual int cleanSignalConnect(QString signame, QString sltName, QString objectName, QString actualSlot) = 0;
};

#endif // __ARCSQDYNAMICOBJECT_H__
