/*
  name: include/arcs/arcsprocess.h

  This file is part of ARCS - Augmented Reality Component System
  (version 2-current), written by Jean-Yves Didier 
  for IBISC Laboratory (http://www.ibisc.univ-evry.fr)

  Copyright (C) 2013  Universit d'Evry-Val d'Essonne

  This program is free software: you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program.  If not, see <http://www.gnu.org/licenses/>.


  Please send bugreports  with examples or suggestions to
  jean-yves.didier__at__ibisc.univ-evry.fr
*/


#ifndef __ARCSPROCESS_H__
#define __ARCSPROCESS_H__

#include <arcs/arcsdll.h>
#include <arcs/arcssheet.h>
#include <arcs/arcs.h>
#include <arcs/arcscontrollercomponent.h>


#include <QMutex>
#include <QWaitCondition>

class ARCSContext;
class ARCSControllerComponent;
class ARCSAppMode;

//! This class is representing a process under ARCS
/*!
* A process manages a set of sheets using a controller.
* \author Jean-Yves Didier
  * \date April, the 5th, 2009
  * \ingroup lib
  */
class DLL_POINT ARCSProcess : public QObject
{
     Q_OBJECT

public:
     ARCSProcess();
     ~ARCSProcess();

     /*! Links this process with the context of the application */
     void setContext(ARCSContext* ctx) { context = ctx; }
     /*! Sets the controller to use with this process */
     bool setController(QString s=QString::null);

     /*! Adds a sheet to the process
       * \param s the name of the sheet
       * \param sh the actual sheet structure
       */
     void addSheet(QString s, ARCSSheet sh);
     /*! Removes a sheet from the sheet list of the process
       * \param s the name of the sheet to remove
       */
     void removeSheet(QString s);

    /*! Renames a sheet in the process list
      * This function will do nothing if newName is already taken by another sheet.
      * \param oldName old sheet name
      * \param newName new sheet name
      */
     void renameSheet(QString oldName, QString newName);

     /*! Returns the sheet with the given name
       * \param s the name of the sheet to return
       * \return a sheet structure
       */
     ARCSSheet& getSheet(QString s)
     {
          if (sheets.contains(s))
               return sheets[s];
          return ARCSSheet::null ;
     }

     /*! Returns the list of the sheets stored inside the process */
     QStringList getSheetNames() { return sheets.keys(); }
     /*! Returns the id of the controller */
     QString getControllerId()
     {
         if (controller)
             return controller->getProperty("id").toString();
         return QString::null;
     }

     /*! Sets the application mode of the process
       * This method is intended to be used with the main process of the application.
       * For other process, you should use setReferenceApplicationMode()
       * \param flag the application mode flag to use.
       * \sa setReferenceApplicationMode()
       */
     void setApplicationMode(ARCS::ARCSAppFlag  flag) { currentMode = flag; setProcessMode();}
     /*! Sets the reference application mode the process
       * Use this method when the process is not the main process of the application.
       * This method will then compute the resulting application mode for this process
       * i.e. if it is a basic thread or a thread with an event loop.
       * \param flag the reference application mode flag to use
       */
     void setReferenceApplicationMode(ARCS::ARCSAppFlag flag);

     /*! starts the process */
     bool startProcess();
     /*! stops the process (not implemented yet) */
     bool stopProcess();

     /*! An event handler inherited from QObject
       * The purpose is to handle events when the process has an event loop.
       * This is usually triggered by the ARCSAppMode::tokenHook() if needed.
       */
     virtual bool event(QEvent* event);

     /*! Waits for the first sheet of the process to be set up
       * This method is needed because subprocesses must be initialized before the main process starts.
       */
     void waitForFirstSheet();

     /*! Returns the actual controller component */
     ARCSControllerComponent* getController() { return controller;}

     /*! Returns the context to which this process is linked */
     ARCSContext* getContext() { return context; }

     /*! Tells wether the process has finished its task or not (not thread-reentrant) */
     bool hasFinished() { return processFinished; }

     /*! Sets up the next sheet */
     void setupNextSheet();

     /*! Waits for the end of this process */
     void wait();


public slots:
     /*! Changes the current sheet
       * This is usually triggered by the controller.
       * The current sheet is deactivated and setupNextSheet() is eventually triggered.
       * \param s the name of the new sheet.
       */
     void changeSheet(QString s);
     /*! Trigger this slot to finish the process (for internal use only) */
     void finish();

signals:
     /*! Triggered when the process is finished */
     void finished();


private:
     void setProcessMode();
     ARCSContext* context;
     QHash<QString,ARCSSheet> sheets;
     ARCSControllerComponent* controller;
     ARCSAppMode* applicationMode;

     bool processFinished;
     bool firstTime;
     ARCS::ARCSAppFlag currentMode;
     QString currentSheetId;
     QMutex mutex;
     QWaitCondition cond;
};


#endif // __ARCSPROCESS_H__
