/*
  name: include/arcs/arcsarray.h

  This file is part of ARCS - Augmented Reality Component System
  (version 2-current), written by Jean-Yves Didier 
  for IBISC Laboratory (http://www.ibisc.univ-evry.fr)

  Copyright (C) 2013  Universit d'Evry-Val d'Essonne

  This program is free software: you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program.  If not, see <http://www.gnu.org/licenses/>.


  Please send bugreports  with examples or suggestions to
  jean-yves.didier__at__ibisc.univ-evry.fr
*/


#ifndef __ARCSARRAY_H__
#define __ARCSARRAY_H__

#include <arcs/arcsfactory.h>
#include <cassert>

/*! \file arcsarray.h
 * \brief This file defines basic templates in order to make arrays of new types available to the ARCS engine.
 *
 * \author Jean-Yves Didier
 * \date October, the 5th, 2009
 * \ingroup lib
 */



//! Template class to implement arrays
/*! \author Jean-Yves Didier
  * \date October, the 5th, 2009
  * \ingroup lib
  */
template<class X> class ARCSArrayTemplate
{
     public:
     ARCSArrayTemplate(const ARCSArrayTemplate& ref)
     {
          size = ref.size;
          data = 0;
          allocate();
          for (unsigned int i  = 0; i < size; i++)
               data[i] = ref.data[i];
     }

     ARCSArrayTemplate() { size = 0; data = 0; }
     ARCSArrayTemplate(unsigned int s) { size = s; allocate(); }
     ~ARCSArrayTemplate() { if (data) delete[] data; }

     void setSize(unsigned int s)
     {
          unsigned int oldSize = size;
          unsigned int i;
          X* oldData = data;
          data = 0;
          allocate();
          if (s < oldSize && oldData != 0)
          {               
               for (i = 0; i < s; i++)
                    data[i] = oldData[i];
          }
          else
          {
               for (i = 0; i < oldSize; i++)
                    data[i] = oldData[i];

          }
          size = s;
          delete[] oldData;
          //allocate();
     }
     unsigned int getSize() { return size; }

     X& operator[] (const unsigned int i) { assert(i < size); return data[i]; }

     ARCSArrayTemplate<X>& operator=(const ARCSArrayTemplate<X>& ref)
                 {
          size = ref.size;
          data = 0;
          allocate();
          for (unsigned int i  = 0; i < size; i++)
               data[i] = ref.data[i];
          return *this;
     }


     private:
     bool allocate() { if ( data) delete[] data; data = new X[size]; return data != 0; }
     unsigned int size;
     X* data;
};



//! Template class to implement array serialization
/*! Factories for arrays should usually derive from this class.
  * If you use the automated process to make your libraries,
  * the name of your class should be <tt>ARCSTypeFactoryTemplate_&lt;yourtypename&gt;Array</tt>.
  * As an example, here is how the internal type of intArray is partially implemented :
  * \code
  template class ARCSArrayTemplate<int>;
  typedef ARCSArrayTemplate<int> intArray;
  Q_DECLARE_METATYPE(intArray);
  class ARCSTypeFactory_intArray : public ARCSArrayFactoryTemplate<int>
  {
  protected:
     QString baseType() const { return "int"; }
  };
  \endcode
  *
  *
  *
  * \author Jean-Yves Didier
  * \date October, the 5th, 2009
  * \ingroup lib
  */

template<class X> class ARCSArrayFactoryTemplate : public ARCSTypeFactory
{
     public:
     virtual QString getTypeName() const { return baseType() + "Array";}

     virtual QString toString(QVariant v);
     virtual QVariant parseString(QString s);

     protected:
     virtual QString separator() const { return ":" ; }
     virtual QString baseType() const = 0;
};



template<class X>
QString  ARCSArrayFactoryTemplate<X>::toString(QVariant v)
{
     if (! v.canConvert<ARCSArrayTemplate<X> >())
          return QString::null;

     ARCSFactory* factory = ARCSFactory::getInstance();
     if (!factory)
          return QString::null;


     QString res;
     ARCSArrayTemplate<X> array = v.value<ARCSArrayTemplate<X> >();
     for (unsigned int i = 0; i < array.getSize() ; i++)
     {
          QVariant v = QVariant::fromValue(array[i]);
          if (i !=0  )
               res+= separator();
          res += factory->dataSerialize(v);
     }
     return res;
}

template<class X>
          QVariant ARCSArrayFactoryTemplate<X>::parseString(QString s)
{     
     ARCSArrayTemplate<X> array;
     if (s.isEmpty())
          return QVariant::fromValue(array);

     ARCSFactory* factory = ARCSFactory::getInstance();
     if (!factory)
          return QVariant();

     QStringList lst = s.split(separator());

     array.setSize(lst.count());

     for ( int i = 0; i < lst.count() ; i++)
     {

     QVariant var = factory->dataDeserialize(baseType(), lst[i]);
     if (var.canConvert<X>())
          array[i] =  var.value<X>();

          }
     return QVariant::fromValue(array);
}

#endif // __ARCSARRAY_H__
