/*
  name: include/arcs/arcsabstractcomponent.h

  This file is part of ARCS - Augmented Reality Component System
  (version 2-current), written by Jean-Yves Didier 
  for IBISC Laboratory (http://www.ibisc.univ-evry.fr)

  Copyright (C) 2013  Universit d'Evry-Val d'Essonne

  This program is free software: you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program.  If not, see <http://www.gnu.org/licenses/>.


  Please send bugreports  with examples or suggestions to
  jean-yves.didier__at__ibisc.univ-evry.fr
*/


#ifndef __ARCSABSTRACTCOMPONENT_H__
#define __ARCSABSTRACTCOMPONENT_H__

#include <arcs/arcsdll.h>

#include <QVariant>
#include <QStringList>
#include <QObject>
#include <QMetaMethod>
#include <QList>


typedef QList<QObject*> ObjectList;
class QMetaObject;

/*! \brief Class handling the generic description of a component.
 *
 * The generic description of a component should contain the following topics 
 * that are adressed in this class:
 * <ul>
 * <li>Instanciation and destruction of the real component (see instanciate(), isInstanciated(), destroy());</li>
 * <li>Signal/slot management (see getSlots(), getSignals());</li>
 * <li>Connections management (see connect(), disconnect(), init(), getProxySlot(), getProxySignal(), genuineConnect(), genuineDisconnect()) ;</li>
 * <li>Serialization/Deserialization (see toString(), parseString());</li>
 * <li>Meta-properties (not directly used by the component but by frameworks using this component).</li>
 * </ul>
 * 
 * For each subclasses of ARCSAbstractComponent, the following methods should be overloaded :
 * <ul>
 * <li> getSlots(), getSignals(); </li>
 * <li> addSlot(), addSignal(), removeSlot(), removeSignal(); </li>
 * <li> toString(), parseString(); </li>
 * <li> getProxySlot(), getProxySignal(); </li>
 * <li> genuineConnect(), genuineDisconnect(); </li>
 * <li> genuineInstanciate(), genuineDestroy(); </li>
 * <li> genuineClone(). </li>
 * </ul>
 *
 * To implement a new component following this guideline, you should have a header like this :
 * \code
 * #ifndef __MYCOMPONENT_H__
 * #define __MYCOMPONENT_H__
 * #include <arcs/arcsabstractcomponent.h>
 *
 *  class MyComponent : public ARCSAbstractComponent
 *  {
 *   public:
 *    MyComponent();
 *    ~MyComponent();
 *
 *    virtual QString toString();
 *    virtual bool parseString(QString s);
 *    virtual QStringList getSignals();
 *    virtual QStringList getSlots();
 *
 * protected:
 *    virtual void getProxySlot (QString slot, ObjectList &obj, QStringList &proxySlot);
 *    virtual void getProxySignal (QString signal, ObjectList &obj, QStringList &proxySignal);
 *    virtual bool genuineConnect (QString sig, ARCSAbstractComponent *dst, QString slt, bool queued=false);
 *    virtual bool genuineDisconnect (QString sig, ARCSAbstractComponent *dst, QString slt);
 *    virtual bool genuineInstanciate ();
 *    virtual void genuineDestroy ();
 *    virtual QVariant getGenuineComponentInstance() { return QVariant(); }
 * };
 *
 * #endif //__MYCOMPONENT_H__
 * \endcode
 *
 * \author Jean-Yves Didier
 * \date October, 2007
 * \ingroup lib
 */

class DLL_POINT ARCSAbstractComponent
{
public:
     ARCSAbstractComponent();
     virtual  ~ARCSAbstractComponent();

     // instanciation / destruction
     /*! \brief Instanciates the real component.
      */
     bool instanciate() { 
       instanciated = genuineInstanciate(); return instanciated;
     }
     /*! \brief Determines wether the component is instanciated or not.
      */
     bool isInstanciated() { return instanciated; } // trouver un mcanisme pour pargner de coder cette fonction
     /*! \brief Destroys the real component.
      */
    void destroy() { if( instanciated) genuineDestroy(); instanciated = false; }

     //! Kage bunshin no jutsu !
     /*! This methods intends to clone an abstract component.
      *  It also takes care of meta-properties like: type, family and so on.
      */
     ARCSAbstractComponent* clone();

     // gestion des signaux et des slots

     /*! \brief Returns the names of the available slots.
      */
     virtual QStringList getSlots() = 0;
     /*! \brief Returns the names of the availables signals.
      */
     virtual QStringList getSignals() = 0;
     

     // connection
     /*! \brief This method connects two components by the indicated signals and slots.
      *
      * In order to lessen the burden of managing connections between components of the same 
      * family, it is possible to implement a direct connection between them in the method
      * genuineConnection(). If it is not possible, the component abstraction must 
      * wrap it to the native format of ARCS components using a QObject intermediate acting 
      * as a proxy (see also getProxySlot() and getProxySignal()).
      * \param sig the signal name of the current component
      * \param dst a pointer to the component to connect to
      * \param slt the slot name of the component to connect to
      * \param queued if set to <tt>false</tt> the parameter transmission will not be delayed.
      * \return <tt>true</tt> if successful, <tt>false</tt> otherwise.
      */
     virtual bool connect(QString sig, ARCSAbstractComponent* dst, QString slt, bool queued = false);

     /*! \brief This method disconnects two components by the indicated signals and slots.
      *
      * In order to lessen the burden of managing connections between components of the same 
      * family, it is possible to implement a direct connection between them in the method
      * genuineDisconnection(). If it is not possible, the component abstraction must 
      * wrap it to the native format of ARCS components using a QObject intermediate acting 
      * as a proxy (see also getProxySlot() and getProxySignal()).
      * \param sig the signal name of the current component
      * \param dst a pointer to the component to connect to
      * \param slt the slot name of the component to connect to
      * \return <tt>true</tt> if successful, <tt>false</tt> otherwise.
      */
     bool disconnect(QString sig, ARCSAbstractComponent* dst, QString slt);

     /*! \brief Initializes a component
      * 
      * Each slot of a component can be used to give values to alter its behaviour.
      * This is the role of this method.
      * \param slt the slot name of this component.
      * \param var a QVariant representing the value to give to this slot.
      */
     bool init(QString slt, QVariant* var);

     // srialisation / dsrialisation
     /*! \brief Defines a basic seralization mechanism.
      * 
      * This should return a string representation of the component.
      */
     virtual QString toString() = 0;

     /*! \brief Defines a basic seralization mechanism.
      * 
      * This should parse a string and affect the internal representation of the component.
      * \return <tt>true</tt> if parsing was a success, <tt>false</tt> otherwise.
      */
     virtual bool parseString(QString s) = 0;


     /*! \brief Defines a way to load a component description from a file.
      *
      * This method has a default implementation. Optimizations might require to rewrite it.
      * \param fn a filename where component description is stored
      * \return <tt>true</tt> if loading and parsing was a success, <tt>false</tt> otherwise.
      */
     virtual bool loadFile(QString fn);

     /*! \brief Defines a way to save a component description inside a file.
      *
      * This method has a default implementation. Optimizations might require to rewrite it.
      * \param fn a filename where component description will be stored.
      * \return <tt>true</tt> if saving a success, <tt>false</tt> otherwise.
      */
     virtual bool saveFile(QString fn);

     // proprits 

     /*! \brief Sets a meta-property on this component.
      * \param name name of the property
      * \param value a QVariant representing the value of the property.
      */
     void setProperty(QString name, QVariant value);

     /*! \brief Gets a meta-property from this component.
      * \param name name of the property
      * \return a QVariant representing the value of the property.
      */
     QVariant getProperty(QString name);

     /*! \brief Removes a meta-property from this component.
      * \param name name of the property
      */
     void removeProperty(QString name);

     /*! \brief Sets the property list to a blank list.
      */
     void resetProperties();
     
     /*! Recovers the list of properties stored inside the component.
      */
     QStringList getProperties();


     /*! Sets the family name to which this component belongs.
      */
     void setFamily(QString s) { family=s; }
     /*! Gets the family name to which this component belongs.
      */
     QString getFamily() { return family; }

     /*! Sets the class name of this component.
      */
     void setType(QString s) { type =s ;}

     /*! Gets the class name of this component.
      */
     QString getType() { return type; }

     /*! Allows to get the genuine component instance.
      *
      * This method purpose is to help in invocating nested objects.
      */
     virtual QVariant getGenuineComponentInstance() { return QVariant(); }

protected:
     /*! \brief Creates a proxy slot to interface a component to native ARCS components.

      *
      * This function should be overriden by subclasses.
      * It is used in the case the abstracted component is not from the same family as another
      * component to connect to.
      * In this case, the abstracted component should offer a wrapper using a QObject to communicate 
      * with other components as if it was a native component of the ARCS runtime.
      * \param slot the slot name of the current component
      * \param obj a list of QObject wrappers returned by this method
      * \param proxySlot the list of the real slot names associated to the QObjects returned by this method.
      */
     virtual void getProxySlot(QString slot, ObjectList & obj, QStringList & proxySlot)=0;

     /*! \brief Creates a proxy signal to interface a component to native ARCS components.
      *
      * This function should be overriden by subclasses.
      * It is used in the case the abstracted component is not from the same family as another
      * component to connect to.
      * In this case, the abstracted component should offer a wrapper using a QObject to communicate 
      * with other components as if it was a native component of the ARCS runtime.
      * \param sig the signal name of the current component
      * \param obj a list of QObject wrappers returned by this method
      * \param proxySignal the list of the real slot names associated to the QObjects returned by this method.
      */     
     virtual void getProxySignal(QString signal, ObjectList & obj, QStringList & proxySignal)=0;

     /*! \brief This method connects two components by the indicated signals and slots.
      *
      * In order to lessen the burden of managing connections between components of the same 
      * family, this method should implement a direct connection between them.
      * \param sig the signal name of the current component
      * \param dst a pointer to the component to connect to
      * \param slt the slot name of the component to connect to
      * \param queued if set to <tt>false</tt> the parameter transmission will not be delayed.
      * \return <tt>true</tt> if direct connection has been performed, <tt>false</tt> otherwise.
      */
     virtual bool genuineConnect(QString, ARCSAbstractComponent*, QString, bool  = false) {return false;}
     /*! \brief This method disconnects two components by the indicated signals and slots.
      *
      * In order to lessen the burden of managing connections between components of the same 
      * family, this method should implement a direct disconnection between them.
      * \param sig the signal name of the current component
      * \param dst a pointer to the component to connect to
      * \param slt the slot name of the component to connect to
      * \return <tt>true</tt> if direct connection has been performed, <tt>false</tt> otherwise.
      */
     virtual bool genuineDisconnect(QString, ARCSAbstractComponent*, QString) {return false;}


     //! This method should implement the instanciation of the actual component.
     /*! Should be overloaded by each subclasses.
      */
     virtual bool genuineInstanciate() = 0;

     //! This method should implement the destruction of the actual component.
     /*! Should be overloaded by each subclasses.
      */
     virtual void genuineDestroy() = 0;


     //! Wrapper for getProxySlot()
     static void getProxySlot(ARCSAbstractComponent* cmp, QString slot, 
			      ObjectList& obj, QStringList & proxySlot)
     {
	  cmp->getProxySlot(slot, obj, proxySlot);
     }

     //! Wrapper for getProxySignal()
     static void getProxySignal(ARCSAbstractComponent* cmp, QString signal, 
			      ObjectList& obj, QStringList & proxySignal)
     {
	  cmp->getProxySignal(signal, obj, proxySignal);
     }




     static const QString slotPrefix;
     static const QString signalPrefix;

private:
     QVariantMap properties;
     QString family;
     QString type;

     bool instanciated;
};


namespace ARCS {
    /** \brief Helper method for components
     *
     * When the core of an ARCSAbstractComponent needs to get list of
     * signals and slots of a QOObject, you can use this method in order
     * to get the complete list.
     * \param object Pointer on a QMetaObject from which you want to get methods
     * \param type Type of methods to return. Use QMetaMethod::Slot for slots and QMetaMethod::Signal for signals
     * \return A QStringList of method names.
     */
    QStringList getMethodList(const QMetaObject *object,QMetaMethod::MethodType type);
}

#endif //__ARCSABSTRACTCOMPONENT_H__


