/*
  name: metaconfigwriter.cpp

  This file is part of ARCS - Augmented Reality Component System,
  written by Jean-Yves Didier, Vincent Le Ligeour and Yoann Petit
  for IBISC Laboratory (http://www.ibisc.univ-evry.fr)

  Copyright (C) 2004 Universit d'Evry-Val d'Essonne

  This program is free software: you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program.  If not, see <http://www.gnu.org/licenses/>.


  Please send bugreports  with examples or suggestions to
  didier__at__iup.univ-evry.fr
*/


#include <metalibrary/metaconfigwriter.h>

#include <qfile.h>
#include <qdir.h>


MetaConfigWriter::MetaConfigWriter(QString ns)
{
     isBlock = !ns.isEmpty();
     nameSpace = ns;

     eDefines = application.createElement("defines");
     eLibraries = application.createElement("libraries");
     eObjects = application.createElement("objects");

     if (isBlock)
     {
	  QDomElement eBlock = application.createElement("block");
	  application.appendChild(eBlock);
	  eBlock.appendChild(eDefines);
	  eBlock.appendChild(eLibraries);
	  eBlock.appendChild(eObjects);
	  eSheet = application.createElement("sheet");
	  eBlock.appendChild(eSheet);
	  eSignals = application.createElement("signals");
	  eBlock.appendChild(eSignals);
	  eSlots = application.createElement("slots");
	  eBlock.appendChild(eSlots);
     }
     else
     {
	  QDomElement eApplication = application.createElement("application");
	  application.appendChild(eApplication);
	  eApplication.appendChild(eDefines);
	  eApplication.appendChild(eLibraries);
	  eApplication.appendChild(eObjects);
	  eSheets = application.createElement("sheets");
	  eApplication.appendChild(eSheets);
	  eStateMachine = application.createElement("statemachine");
	  eStateMachine.setAttribute("terminal","end");
	  eApplication.appendChild(eStateMachine);
     }

     blockList.clear();
}


MetaConfigWriter::~MetaConfigWriter()
{

}


void MetaConfigWriter::reset()
{
     
}


QString MetaConfigWriter::computeRelativePath(QString fp, QString lp)
{
     int numsec=0;

     while( fp.section(QDir::separator(),numsec, numsec+1) == lp.section(QDir::separator(),numsec, numsec+1) )
	  numsec ++;


     QString fp2 = fp.section(QDir::separator(), numsec+1);
     QString lp2 = lp.section(QDir::separator(), numsec+1);

     int idx = 0;
     QString res;

     while( (idx = fp2.find(QDir::separator(),idx+1) ) >= 0 )
	  res = res + "../" ;
     res = res + /*"../" +*/ lp2;
     
     //std::cerr << "Test librairie :  fp :" << fp.ascii() << " lp:" << lp.ascii() << " res:" << res.ascii() << std::endl;
     //std::cerr << "Paths fp2:" << fp2.ascii() << " lp2:" << lp2.ascii() << std::endl;
     return res;
}


void MetaConfigWriter::processLibrariesPath(QString filePath)
{
     if (!eLibraries.hasChildNodes())
	  return;
     
     QDomNodeList nl = eLibraries.childNodes();

     for (uint i=0; i < nl.length() ; i++)
     {
	  QDomNode node = nl.item(i);
	  if ( node.isElement() && (node.nodeName() == "library"))
	  {
	       QDomElement e = node.toElement();
	       e.setAttribute("name"  ,computeRelativePath(filePath,e.attribute("name")));
	  }
     }
}


void MetaConfigWriter::processObjectsPath(QString filePath)
{
     if (!eObjects.hasChildNodes())
	  return;
     
     QDomNodeList nl = eObjects.childNodes();

     for (uint i=0; i < nl.length() ; i++)
     {
	  QDomNode node = nl.item(i);
	  if ( node.isElement() && (node.nodeName() == "object"))
	  {
	       QDomElement e = node.toElement();
	       if ( e.hasAttribute("file"))
		    if (filePath != e.attribute("file"))
			 e.setAttribute("file"  ,computeRelativePath(filePath,e.attribute("file")));
	  }
     }
}


bool MetaConfigWriter::writeFile(QString filename)
{
	std::cout << "[MeCW] Writing down block file " << filename.ascii() << std::endl;
     QFile file( filename );
     if ( !file.open( IO_WriteOnly ) )
     {
	     std::cerr << "[META] couldn't open xml file for saving." << std::endl;
          return false;
     }
     processLibrariesPath(filename);
     processObjectsPath(filename);
     QTextStream ts(&file);
     application.save(ts, 4);
     file.close();

     
     for (uint i=0; i < blockList.count(); i++)
     {
	  QString bn = blockList[i];
	  MetaConfigWriter mcw(bn);
	  if (libraries != NULL)
	       mcw.addLibraries(libraries);
	  mcw.addDefines();
	  mcw.addObjects();

	  mcw.addSheet(MetaSheet::getSheet(bn));
	  // Puis ajouter les signaux et les slots
	  MetaObject* mo = MetaObject::getObject(bn);
	  if (mo != NULL)
	  {
	       if (mo->isBlock())
	       {
		    QDictIterator< MetaIO > itesig = MetaBlock::getSignalList();
		    for( ; itesig.current(); ++itesig)
			 if (itesig.current()->getPublicName().startsWith(bn))
			      mcw.addMethod(itesig.current(), false);
		    
		    QDictIterator< MetaIO > iteslt = MetaBlock::getSlotList();
		    for( ; iteslt.current(); ++iteslt)
		    {
			 if (iteslt.current()->getPublicName().startsWith(bn))
			      mcw.addMethod(iteslt.current(), true);
		    }
	       }
	  }
	  QFileInfo fi(bn);
	  if (fi.isRelative())
	  {
	       fi = QFileInfo(QFileInfo(filename).dirPath() +  QDir::separator() + bn);
	  }
	  mcw.writeFile(fi.filePath());
     }

     return true;
}



void MetaConfigWriter::setTerminalState(QString s)
{
     eStateMachine.setAttribute("terminal", s);
}


void MetaConfigWriter::setInitialState(QString s)
{
     eStateMachine.setAttribute("initial", s);
}



void MetaConfigWriter::addTransition(QString a, QString b, QString token)
{
     QDomElement e = application.createElement("transition");
     e.setAttribute("stepA", a);
     e.setAttribute("token", token);
     e.setAttribute("stepB", b);
     eStateMachine.appendChild(e);
}

void MetaConfigWriter::addDefine(QString id, QString type, QString value)
{
     QDomElement e = application.createElement( "define" );
     e.setAttribute("id", id);
     e.setAttribute("type" , type);
     e.setAttribute("value" , value);
     eDefines.appendChild(e);    
}

void MetaConfigWriter::addDefines()
{
     QDictIterator<MetaDefine> it = MetaDefine::getIterator();

     for ( ; it.current() ; ++it)
     {
	  if (!isBlock)
	       if (!it.current()->getID().contains("::"))
		    addDefine((it.current())->getID(), (it.current())->getType(), (it.current())->getValue());
	  else
	       if (it.current()->getID().startsWith(nameSpace))
		    addDefine((it.current())->getID().section("::",1), (it.current())->getType(), (it.current())->getValue());
     }
}

void MetaConfigWriter::addObject(QString id, QString cn, bool persistent,bool b)
{
     if (cn  == nameSpace )
	  return;


     QDomElement e = application.createElement( "object" );
     e.setAttribute("id" , id);
          
     if (b)
     {
	  e.setAttribute("file", cn);
	  if (!blockList.contains(cn))
	       blockList.append(cn);
     }
     else
	  e.setAttribute("classname", cn);

     if (!isBlock)
	  e.setAttribute("persistent" , (persistent)?"true":"false");		    

     eObjects.appendChild(e);
}



void MetaConfigWriter::addObjects()
{
     QDictIterator<MetaObject> it = MetaObject::getIterator();

     for ( ; it.current() ; ++it)
     {
	  if (!isBlock)
	  {
	       if (!it.current()->getID().contains("::"))
	       {
		    if (it.current()->isBlock())
		    {
			 MetaBlock* mb = (MetaBlock*)it.current();
			 if (mb->getMetaBlockLink() != NULL)
			 {
			      addObject((it.current())->getID(), mb->getMetaBlockLink()->getID(),//((MetaBlock*)it.current())->getFile(),
					(it.current())->isPersistent(),it.current()->isBlock());
			 }
		    }
		    else
			 addObject((it.current())->getID(), it.current()->getClassName(), 
				   it.current()->isPersistent(), it.current()->isBlock());
	       }
	  }
	  else
	  {
	       if (it.current()->getID().startsWith(nameSpace))
	       {
		    if (it.current()->isBlock())
		    {
			 MetaBlock* mb = (MetaBlock*)it.current();
			 if (mb->getMetaBlockLink() != NULL)
			 {
			      addObject((it.current())->getID().section("::",1), mb->getMetaBlockLink()->getID(),
					(it.current())->isPersistent(),it.current()->isBlock());
			 }
		    }
		    else
			 addObject((it.current())->getID().section("::",1), it.current()->getClassName(), 
				   it.current()->isPersistent(), it.current()->isBlock());
	       }
	  }
     }
}


void MetaConfigWriter::addLibraries(MetaLibrary* ml)
{
     QStringList lst = ml->getLibraries();
     libraries = ml;
     
     for (uint i = 0; i < lst.count(); i++ )
     {
	  QDomElement e = application.createElement( "library" );
	  e.setAttribute("name", lst[i]);
	  eLibraries.appendChild(e);
     }
}



void MetaConfigWriter::addMethod(MetaIO* mio, bool slot)
{
     if (mio == NULL)
	  return;

     QDomElement e = application.createElement( "method");
     e.setAttribute("name", mio->getPublicName().section("::",1));
     e.setAttribute("object", mio->getObjectName().section("::",1));
     e.setAttribute("method", mio->getIOObjectName());
     if (slot)
	  eSlots.appendChild(e);
     else
	  eSignals.appendChild(e);
}


void MetaConfigWriter::setTypeAttribute(QDomElement e, int type)
{
     switch (type)
     {
     case MetaInit::Bool:
	  e.setAttribute("type", "bool");
	  break;
     case MetaInit::Integer:
	  e.setAttribute("type", "int");
	  break;
     case MetaInit::Float:
	  e.setAttribute("type", "float");
	  break;
     case MetaInit::Double:
	  e.setAttribute("type", "double");
	  break;
     case MetaInit::String:
	  e.setAttribute("type", "string");
	  break;
     case MetaInit::Void:
	  e.setAttribute("type", "void");
	  break;
     case MetaInit::Object:
	  e.setAttribute("type", "object");
	  break;
     case MetaInit::Define:
	  e.setAttribute("type", "define");
	  break;
     case MetaInit::Path:
	  e.setAttribute("type", "path");
	  break;
     default:
	  e.setAttribute("type", "unknown");
     }
}


void MetaConfigWriter::addSheet(MetaSheet* ms)
{
     QDomElement e;
     int i;
     if (ms->isLinkedToBlock())
	  e = eSheet;
     else
     {
	  e = application.createElement("sheet");
	  e.setAttribute("id", ms->getSheetID());
     }

     QDomElement prec = application.createElement("preconnection");
     QDomElement conn = application.createElement("connection");
     QDomElement post = application.createElement("postconnection");
     
     if ( ms->getPreConnectionsNumber() )
	  e.appendChild(prec);
     if ( ms->getConnectionsNumber() )
	  e.appendChild(conn);
     if ( ms->getPostConnectionsNumber() )
	  e.appendChild(post);
     
     if (!ms->getTokenSender().isEmpty())
     {
	  QDomElement f = application.createElement("tokensender");
	  f.setAttribute("object", (isBlock)?ms->getTokenSender().section("::",1):ms->getTokenSender());
	  e.appendChild(f);
     }

     for ( i = 0; i < ms->getPreConnectionsNumber() ; i++ )
     {
	  QDomElement f = application.createElement("init");
	  MetaInit mi = ms->getPreConnection(i);
	  f.setAttribute("object", (isBlock)?mi.getObject().section("::",1):mi.getObject());
	  f.setAttribute("slot", mi.getSlot());
	  setTypeAttribute(f,mi.getType());
	  f.setAttribute("value", mi.getValue());
	  prec.appendChild(f);
     }
     
     for ( i = 0; i < ms->getConnectionsNumber() ; i++ )
     {
	  QDomElement f = application.createElement("wire");
	  MetaWire mi = ms->getConnection(i);
	  f.setAttribute("objsource", (isBlock)?mi.getSender().section("::",1):mi.getSender());
	  f.setAttribute("signal", mi.getSignal());
	  f.setAttribute("objdest", (isBlock)?mi.getReciever().section("::",1):mi.getReciever());
	  f.setAttribute("slot", mi.getSlot());
	  conn.appendChild(f);
     }

     for ( i = 0; i < ms->getPostConnectionsNumber() ; i++ )
     {
	  QDomElement f = application.createElement("init");
	  MetaInit mi = ms->getPostConnection(i);
	  f.setAttribute("object", (isBlock)?mi.getObject().section("::",1):mi.getObject());
	  f.setAttribute("slot", mi.getSlot());
	  setTypeAttribute(f,mi.getType());
	  f.setAttribute("value", mi.getValue());
	  post.appendChild(f);
     }

     if (!ms->isLinkedToBlock())
//	  eBlock.appendChild(e);
//     else
	  eSheets.appendChild(e);

}



void MetaConfigWriter::addSheets()
{
     QDictIterator<MetaSheet> it = MetaSheet::getIterator() ;

     for ( ; it.current() ; ++it)
	  if ( ! (it.current()->isLinkedToBlock()) )
	       addSheet(it.current());
}
