/*
  name: metabase.h

  This file is part of ARCS - Augmented Reality Component System,
  written by Jean-Yves Didier, Vincent Le Ligeour and Yoann Petit
  for IBISC Laboratory (http://www.ibisc.univ-evry.fr)

  Copyright (C) 2004 Universit d'Evry-Val d'Essonne

  This program is free software: you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program.  If not, see <http://www.gnu.org/licenses/>.


  Please send bugreports  with examples or suggestions to
  didier__at__iup.univ-evry.fr
*/


#ifndef __METABASE_H__
#define __METABASE_H__

#include <qmap.h>
#include <qdict.h>
#include <qobject.h>
#include <vector>
#include <iostream>
#include <qstringlist.h>
#include <metalibrary/metawin.h>

//using namespace std;

#define SLOT_PREFIX "1"
#define SIGNAL_PREFIX "2"


// Compatibility with Visual C++ 6
#ifdef WIN32
class MetaObject;
class MetaDefine;
class MetaWire;
class MetaInit;
class MetaSheet;
class MetaIO;

//template class DLL_POINT std::vector<MetaWire>;
//template class DLL_POINT std::vector<MetaInit>;
//template class DLL_POINT QDict<MetaObject>;
//template class DLL_POINT QDict<MetaDefine>;
//template class DLL_POINT QDict<MetaSheet>;
//template class DLL_POINT QDict<MetaIO>;
#endif //WIN32


/*! \file metabase.h
 * \author Jean-Yves Didier
 * \date 06 Juin 2005
 * 
 * This file is describing all the basic elements we need to make the runtime
 * work. These elements are also used and generated by the XML parser which is 
 * described in class MetaConfigReader.
 */

class MetaObject;
template class DLL_POINT QDict<MetaObject>;


/*! \brief A class describing an object as in the application dtd
 * 
 * An object is basically referenced by a name, a class name and a reference 
 * to the object.
 */
class DLL_POINT MetaObject
{
     typedef QDict<MetaObject> MetaObjectList;

public:
     /*! \brief Construtor for a MetaObject
      *
      * \param cn gives a classname
      * \param i gives an object ID
      * \param p gives the persistent attribute (<TT>true</TT> by default)
      */
     MetaObject(QString cn, QString i, bool p = true);
     virtual  ~MetaObject();
     
     /*! \brief Allows to retrieve the counterpart of the MetaObject, ie the instanciated QObject
      * 
      * \return a pointer to the QObject or NULL if not instanciated yet.
      * \sa setReference(QObject* o)
      */
     QObject* getReference() { return reference; }
     
     /*! \brief Returns the object's ID
      * \return the object's ID
      */
     QString getID() { return objectID; }

     /*! \brief Retrieve the persistent behaviour of the object.
      * 
      * The persistent behaviour is indicating if the object will need to be instanciated 
      * once and for all during the time the application will run or if it will be instanciated
      * on demand.
      * \return true if the object is persistent, false otherwise.
      */
     bool isPersistent() {return persistent; }

     /*! \brief Changes the presistence attribute of the object.
      * \return the persistence value as a boolean <tt>true</tt> if persistent, <tt>false</tt> otherwise.
      */
     bool switchPersistence() { persistent = (!persistent); return persistent; }

     /*! \brief says wether the object is indeed a macro-block , that is to say a block composed of several objects or not
      *
      * \return true if the object is a macro-block, <tt>false</tt> otherwise.
      */
     virtual bool isBlock() { return false; }

	/*!
	 * \brief tells the object has been processed or not.
	 * This is mainly usefull for graphical editor
	 */	
     void markAsProcessed() { processed = true; }

     
     /*! \brief says wether the object has been processed or not.
      * 
      * \return true if the object has been processed <tt>false</tt> otherwise
      */
     bool isProcessed() { return processed; } 


     /*! \brief Set the reference associated with the real Object.
      *
      * \param o Reference to QObject.
      * \sa getReference()
      */
     void setReference(QObject* o) { reference = o; }
     /*! \brief Get the class name of the MetaObject.
      *
      * \return The class name of the MetaObject.
      */
     INLINE_MACRO (QString getClassName() , return className; )

     /*! \brief Returns the number of MetaObject objects instanciated.
      * \return The number of MetaObject objects instanciated.
      */
     static int getMetaObjectNumber() { return mol.count(); }
     /*! \brief Returns a pointer to one of the MetaObject objects instanciated.
      * \param s The object name.
      * \return A pointer to the MetaObject named \a s
      */
     static INLINE_MACRO (MetaObject* getObject(QString s) , return mol[s]; )

     /*! \brief Returns an iterator on the MetaObject's dictionnary
      * \return an iterator on the MetaObject's dictionnary
      */
     static INLINE_MACRO ( QDictIterator<MetaObject> getIterator() , return QDictIterator<MetaObject>(mol); )
     /*! \brief Flushes the dictionnary of all created MetaObject so far.
      */
     static void clear();

     /*! \brief Strips the namespace from the complete name of the object.
      * \return The name of the object stripped from its namespace.
      */
     static QString stripNamespace(QString s) { QString tmp = s.section("::",1,1); return (tmp.isEmpty()?s:tmp); }
     
//private:
     DLL_POINT friend std::ostream& operator<<(std::ostream& os, const MetaObject& mo);

protected:
     QString objectID;
     QString className;
     
private:
     void registerObject(); 
     QObject* reference;
     bool persistent;
     bool processed;
     static QDict<MetaObject> mol;
};

//! A basic type to define a dictionnary of MetaObject
typedef QDict<MetaObject> MetaObjectList;


/*! \brief A class added for macro-blocks support
 *
 * A MetaIO is used to define inputs/outputs of macro-blocks.
 * It's binding macro-block signals/slots with internal representation, ie with
 * internal objects signals/slots.
 */
class DLL_POINT MetaIO
{
public:
     /*! \brief Empty constructor
      */
     MetaIO() : publicName(0) {}
     /*! \brief Constructor
      *
      * \param pn gives the public name of the input/output
      * \param obj gives the internal object ID to which this input/output refers.
      * \param ion gives the internal signal/slot name of the object refered by  object ID.
      */
     MetaIO(QString pn,QString obj, QString ion) ;

     /*! \brief method for compatibility with empty constructor.
      * 
      * \return <tt>true</tt> if this object was created using empty contructor, <tt>false</tt> otherwise.
      */
     bool isEmpty() { return publicName.isEmpty(); }
     
     /*! 
      * \return the public name of the macro-block input/output
      */
     QString getPublicName() { return publicName; }

     /*!
      * \return the internal object ID to which this input/output refers
      */
     QString getObjectName() { return objectName; }
     /*!
      * \return the internal signal/slot of object given by getObjectName() to which this input/output is bound.
      */
     QString getIOObjectName() { return IOObjectName; }

private:
     QString publicName;
     QString objectName;
     MetaObject* object;
     QString IOObjectName;

};

/*! \brief A class describing what MetaDefine(s) are.
 *
 * A MetaDefine almost has the same role as a #define compiler directive 
 * in C or C++ language. This is something that will interpreted 
 * when the xml file will be read. A MetaDefine can be used as a substitute 
 * for MetaInit parameters.
 */
class DLL_POINT MetaDefine
{
typedef QDict<MetaDefine> MetaDefineList;

public:
/*! \brief Constructor.
 *
 * \param i gives define's identifier.
 * \param t gives the type of the init that will be fetched using this define.
 * \param val gives the value of the init that will be fetchd using this define.
 */
     MetaDefine(QString i, QString t, QString val);
     ~MetaDefine();
     /*! \brief Returns the define ID 
      * 
      * \return this define's ID
      */
     QString getID() { return id; }
     /*! \brief Returns this define's value
      *
      * \return the value of this define
      */
     QString getValue() { return value;}
     /*! \brief Gets the overall number of defines created.
      *
      * \return the overall number of defines created.
      */
     static int getMetaDefineNumber() { return mdl.count(); }
     /*! \brief Allows to retrieve a define by its ID.
      *
      * \param s the define's ID
      * \return a pointer to the corresponding MetaDefine object.
      */
     static MetaDefine* getDefine(QString s) { return mdl[s]; }

     /*! \brief Returns the type of this define
      *
      * \return the type of this define
      */
	 INLINE_MACRO( QString getType() , return type ; )

	/*! \brief Returns an iterator on the MetaDefine's dictionnary
      * \return an iterator on the MetaDefine's dictionnary
      */
     static INLINE_MACRO( QDictIterator<MetaDefine> getIterator() , return QDictIterator<MetaDefine>(mdl); )

     /*! \brief Flushes the dictionnary of all created MetaDefine so far.
      */
     static void clear();

private:
     void registerDefine();
     QString id;
     QString type;
     QString value;
     static MetaDefineList mdl;

};

typedef QDict<MetaDefine> MetaDefineList;


/*! \brief A class describing a connection beetween two objects.
 * 
 * A Wire is a connection beetween 2 objects using the signal/slot paradigm.
 * That's why a wire as an object sender, a signal name, an object reciever and 
 * a slot name as attributes.
 */
class DLL_POINT MetaWire
{
public:
     /*! \brief Constructor
      * 
      * \param objA the signal sender object ID
      * \param sigName the signal's name sent by object objA
      * \param objB the slot reciever object ID
      * \param sltName the slot's name sent by object objB
      */
     MetaWire(QString objA, QString sigName, QString objB, QString sltName);
     /*! \brief Retrieves the sender
      *
      * \return the sender's objetc ID
      */
     QString getSender() {return objectSender; }
     /*! \brief Retrieves the receiver 
      *
      * \return the reciever's object ID
      */
     QString getReciever() { return objectReciever; }

     /*! \brief Returns the pointer to the sender object.
      * This function is created for caching purposes.
      * If you just want the name of the sender object then use getSender()
      * \return a pointer to the sender object.
      * \sa getRecieverObject()
      */
     MetaObject* getSenderObject() { return sender; }

     /*! \brief Returns the pointer to the reciever object.
      * This function is created for caching purposes.
      * If you just want the name of the reciever object then use getReciever()
      * \return a pointer to the reciever object.
      * \sa getSenderObject()
      */
     MetaObject* getRecieverObject() { return receiver; }
     /*! \brief Gives the signal name
      *
      * \return the signal name of the sender involveed into connection
      */
     QString getSignal() { return signalName; }

     /*! \brief Gives the real signal name
      *
      * This function is created for caching purposes.
      * Real signal names are handled by Qt by adding a "2" before the public signal name.
      * \return the real signal name of the sender involved into connection
      * \sa getRealSlot()
      */
     QString getRealSignal() { return realSignalName; }
     /*! \brief Gives the slot name
      * 
      * \return the slot name of the reciever involved into connection
      */
     QString getSlot() { return slotName; }
     /*! \brief Gives the real slot name
      *
      * This function is created for caching purposes.
      * Real slot names are handled by Qt by adding a "1" before the public slot name.
      * \return the real slot name of the reciever involved into connection
      * \sa getRealSignal()
      */
     QString getRealSlot() { return realSlotName; }

//private:
     DLL_POINT friend std::ostream& operator<<(std::ostream& os, const MetaWire& mw);
     bool operator<(const MetaWire& mw);//!< Defined because Visual C++ 6 wants it


protected:
     QString objectSender; 
     QString objectReciever;
     QString signalName;
     QString slotName;
    
private: 
     MetaObject* sender;
     MetaObject* receiver;
     QString realSignalName;
     QString realSlotName;
};


/*! \brief A class describing an initialisation.
 *
 * Each object may require a special initialisation.
 * An initialisation can be set up at two different moments :
 * One before the connections are established beetween objects,
 * the other one after the connections are established, allowing 
 * the initialisations to propagate through the objects.
 * 
 * An initialisation is concerning only simple slots with only one parameter 
 * which type should belong to a known list of simple types.
 */
class DLL_POINT MetaInit
{
/*! \todo A template for this class may be suitable in some ways.
 */
public:

     /*! \brief List of allowed initialisation types.
      */
     enum Types{
	  Bool,  //!< For boolean types
	  Integer,  //!< For integer types
	  Float,  //!< For Simple precision floating point numbers
	  Double, //!< For Double precision floating point numbers
	  String, //!< For String types
	  Void, //!< For void types, menaing there's nothing
	  Object, //!< For Object type. If set, then the value is an object's ID (MetaObject).
	  Define, //!< For Define type. If set, then the value is a define's ID (MetaDefine).
	  Path, //!< For Path data.
	  Undefined //!< If the type found was unknown.
     };
     /*! \brief Constructor
      *
      * \param obj The object's ID onto which one will be set up the initialization.
      * \param slt The slot name of the object referred by obj
      * \param t The type as a string
      * \param v The value parsed a s a string.
      */
     MetaInit(QString obj,QString slt, QString t, QString v, bool editMode=false);

     ~MetaInit();
     /*! \brief Returns the type of the initialisation
      *
      * \return  the type of initialisation
      * \sa Types
      */
     int getType() { return type; }
     /*! \brief Returns the object's ID onto which one will be set up the initialization.
      *
      * \return The object's ID onto which one will be set up the initialization.
      */
     QString getObject() { return objectName; }
     /*! \brief Returns the initialization's value
      *
      * \return the initialization's value
      */
     MetaObject* getObjectP() { return object;}
     
     /*! \brief Gives a pointer to where the value in its internal representation is located
      */
     void* getValueData() { return valueData; }

     /*! \brief Gives the string representing the value of initialization
      * \return A string representing initialization value.
      */
     QString getValue() { return value; }
     /*! \brief Set the intrinsic value of initialization parameter
      */ 
     void setValue(QString str) { value=str; } 
     /*! \brief Returns the slot of the object referred by getObject() 
      *
      * \return the slot of the object referred by getObject()
      */
     QString getSlot() { return slotName; }

     /*! \brief Gives the real slot name
      *
      * This function is created for caching purposes.
      * Real slot names are handled by Qt by adding a "1" before the public slot name.
      * \return the real slot name of the reciever involved into connection
      */
     QString getRealSlot() { return realSlotName; }

     void convertValue(); //!< converts the string value representation to an internal representation.

//private:
     DLL_POINT friend std::ostream& operator<<(std::ostream& os, const MetaInit& mi);
     bool operator<(const MetaInit& mi); //!< Defined because Visual C++ 6 wants it
     

protected:
     QString objectName;
     QString value;
     int type;
     
private:
     static int convertStringType(QString t);
     void* valueData;
     MetaObject* object;
     QString slotName;
     QString realSlotName;
};

//! A basic type to define a list of MetaWire
typedef std::vector<MetaWire> MetaWireList;
//! A basic type to define a list of MetaInit
typedef std::vector<MetaInit> MetaInitList;

/*! \brief A class describing a sheet.
 *
 * A Sheet is what we can call a state in our application.
 * It describes the objects working in this state and the relations beetween them,
 * i.e. how they are connected to each other. 
 * A sheet is also providing a way to initialize objects before they're connected to each other,
 * and also after they're connected.
 */
class DLL_POINT MetaSheet
{
     typedef QDict<MetaSheet> MetaSheetList;
public:
     /*! \brief Sheet constructor
      *
      * \param id Gives the sheet ID
      */ 
     MetaSheet(QString id);
     ~MetaSheet();

     /*! \brief Returns the sheet ID
      *
      * \return  the sheet ID
      */
     QString getSheetID() { return sheetID;}
     /*! \brief Returns the number of connections this sheet is describing
      *
      * \return the number of connections this sheet is describing
      * \sa getConnection(int n) and addConnection(MetaWire mw)
      */
     int getConnectionsNumber() { return wires.size(); }
     /*! \brief Gives the n-th connection. 
      *
      * \return The n-th connection.
      * \sa getConnectionsNumber() and addConnection(MetaWire mw)
      */ 
     MetaWire getConnection(int n) { return wires[n]; }
     /*! \brief Adds a connection to the sheet
      *
      * \param mw a MetaWire representing the connection beetween 2 objects.
      */
     void addConnection(MetaWire mw) { wires.push_back(mw); }
     /*! \brief Gives the number of initializations to make before connecting all objects
      * \return  The number of initializations to make before connecting all objects
      */
     int getPreConnectionsNumber() { return preconnections.size(); } 
     /*! \brief Gives the number of initializations to make after connecting all objects
      * \return  The number of initializations to make after connecting all objects
      */
     int getPostConnectionsNumber() { return postconnections.size(); }
     /*! \brief Gives the n-th pre-connection.
      * \return The n-th pre-connection.
      * \sa addPreConnection(MetaInit mi) and getPreConnection(int n)
      */
     MetaInit getPreConnection(int n) { return preconnections[n]; }
     /*! \brief Gives the n-th post-connection.
      * \return The n-th post-connection.
      * \sa addPreConnection(MetaInit mi ) and getPreConnectionsNumber()
      */
     MetaInit getPostConnection(int n) { return postconnections[n]; }
     /*! \brief Adds a pre-connection to the sheet
      *
      * \param mi a MetaInit representing the specific initialisation
      * \sa getPreConnection(int n) and getPreConnectionsNumber()
      */
     
     /*! \brief Tells that a sheet is describing a macro-block.
      */
     void setLinkedToBlock() { blockLink = true; }

     /*! \brief Tells wether a sheet is a part of a block or not.
      * 
      * \return true if linked to a block, false otherwise.
      */
     bool isLinkedToBlock() { return blockLink; }

     /*! \brief Adds a pre-connection to the sheet
      *
      * \param mi a MetaInit representing the specific initialisation
      * \sa getPreConnection(int n) and getPreConnectionsNumber()
      */
     void addPreConnection(MetaInit mi) { preconnections.push_back(mi); }
     /*! \brief Adds a post-connection to the sheet
      *
      * \param mi a MetaInit representing the specific initialisation
      * \sa getPostConnection(int n) and getPostConnectionsNumber()
      */
     void addPostConnection(MetaInit mi ) { postconnections.push_back(mi); }
     /*! \brief Returns the token sender 
      * The token sender is the only object of the sheet allowed to communicate with the state machine.
      * A token sender <b>must possess<b> a sendToken(QString) signal.
      *
      * \return the token sender object ID.
      * \sa setTokenSender(QString s)
      */
     QString getTokenSender() { return tokenSender; }
     /*! \brief Sets the token sender
      *
      * \param s The object ID of the token sender.
      * \sa getTokenSender()
      */
     void setTokenSender(QString s) { tokenSender = s; }


     /*! \brief Returns the number of MetaSheet objects instanciated.
      * \return The number of MetaSheet objects instanciated.
      */
     static int getMetaSheetNumber() { return msl.count(); }
     
     /*! \brief Returns a pointer to one of the MetaSheet objects instanciated.
      * \param s The sheet name.
      * \return A pointer to the MetaSheet named \a s
      */
     static MetaSheet* getSheet(QString s) { return msl[s]; }

     /*! \brief Returns an iterator on the MetaSheet's dictionnary
      * \return an iterator on the MetaSheet's dictionnary
      */
     static QDictIterator<MetaSheet> getIterator() { return QDictIterator<MetaSheet>(msl);}

     /*! \brief Flushes the dictionnary of all created MetaSheet so far.
      */
     static void clear();

//private:
     DLL_POINT friend std::ostream& operator<<(std::ostream& os, const MetaSheet& ms);
     

protected:
     QString sheetID;
     

private:
     void registerSheet();  
     MetaWireList wires;
     MetaInitList preconnections;
     MetaInitList postconnections;
     QString tokenSender;
     bool blockLink;

     static MetaSheetList msl;

};


//! A basic type to define a dictionnary of MetaSheet
typedef QDict<MetaSheet> MetaSheetList;

/*! \brief class describing a macro-block ie an object gathering several objects
 *
 * A Macro-block is an object which could contain several objects inside.
 * For the end-user, it is seen as a real object, while it is only a convenient way to group 
 * several objects (including other macro-blocks) together.
 * This class is inheriting from MetaObject for compatibility purpose.
 */
class DLL_POINT MetaBlock : public MetaObject
{
public:
     /*! \brief Constructor
      * 
      * \param id the macro-block ID.
      * \param p the persistent attribute.
      */
     MetaBlock(QString id, bool p=true); // : MetaObject(id, QString::null, p) {}
     //~MetaBlock();


     /*! \brief says wether the object is indeed a macro-block , that is to say a block composed of several objects or not
      *
      * \return true if the object is a macro-block, <tt>false<tt> otherwise.
      */
     bool isBlock() { return true; }

     /*! \brief Build a link between this metablock and another one.
      * 
      * This is a quite complex mechanism created for edition purpose only.
      * It allows to make several ghost instances of one real macro instance.
      * This real instance is the only one on which modifications are applied during edition.
      * \param mb a pointer to the real instance of the considered macro-block
      */
     void linkToMetaBlock(MetaBlock* mb) { block = mb; mb->representsFile();}

     /*! \brief Gets the potentially existing macro-block linked with this one.
      * 
      * \sa linkToMetaBlock()
      * \return <tt>NULL</tt> if not linked with another macro-block, a pointer to the instance of the macro-block to which it is linked otherwise.
      */
     MetaBlock* getMetaBlockLink() { return block; }

     /*! \brief Seems to be deprecated 
      * \todo investigate if really deprecated
      */
     bool isRepresentingFile() { return fileRepresentation;}
     /*! \brief Seems to be deprecated 
      * \todo investigate if really deprecated
      */
     void representsFile() { fileRepresentation = true;}

     /*! \brief Sets the filename which stores the XML description of the macro-block.
      * \param s String reprenseting the filename.
      */
     void setFile(QString s) { filePath = s; }
     /*! \brief Returns the filename which stores the XML description of the macro-block.
      * \return a string representing a filename.
      */
     QString getFile() { return filePath;}

     //void addSignal(MetaIO sig);// { siglist.insert(sig->getPublicName(), sig); }

     /*! \brief Adds a signal ie a MetaIO which is here an output.
      *
      * \param sig a pointer to the MetaIO describing the signal.
      * \sa getSignal(QString sig)
      */
     void addSignal(MetaIO* sig);
     /*! \brief Adds a slot ie a MetaIO which is here an input.
      *
      * \param slt a pointer to the MetaIO describing the slot.
      * \sa getSlot(QString slt)
      */
     void addSlot(MetaIO* slt);// { sltlist.insert(slt->getPublicName(), slt); }
     /*! \brief Sets the associated sheet to this macro-block
      * 
      * We said a macro-block is a group of objects. These objects can also communicate together.
      * This is exactly the decription of a sheet. That's why the internals of the macro-block is represented as a sheet.
      *
      * \param ms a pointer to a MetaSheet object representing the internal structure of our macro-block
      * \sa getSheet()
      */
     void setMetaSheet(MetaSheet* ms) { sheet = ms; }
     /*! \brief Adds an object to the list of internal objects.
      *
      * \param id the object ID.
      * \sa getObjects()
      */
     void addObject(QString id) { objects.append(id); }
     /*! \brief Tells if the macro-block has been instanciated or not.
      *
      * It means that all the internal objects and the communications beetween them has been instanciated.
      * \param b Set it to <tt>true</tt> if macro-block is instanciated, <tt>false<tt> otherwise.
      * \sa  isInstanciated()
      */
     void setInstanciated(bool b) { instanciated = b; }
     /*! \brief Tells if the macro-block has been instanciated or not.
      *
      * \return <tt>true</tt> if macro-block is instanciated, <tt>false<tt> otherwise.
      * \sa setInstanciated(bool b)
      */
     bool isInstanciated() { return instanciated; }
     /*! \brief Gives the list of the internal objects composing the macro-block.
      *
      * \return the list of the internal objects composing the macro-block.
      * \sa addObject(QString id)
      */
     QStringList getObjects() { return objects; }
     /*! \brief Returns the sheet describing the internals of this macro-block
      *
      * \return a pointer to a MetaSheet
      * \sa setMetaSheet(MetaSheet* ms)
      */
     MetaSheet* getSheet() { return sheet; }

     /*! \brief Gives a MetaIO according to its public name
      *
      * \param sig a public name of an output
      * \return pointer to the MetaIO referred by public name sig
      * \sa addSignal(MetaIO* sig)
      */
     MetaIO* getSignal(QString sig) { return siglist[sig]; }
     /*! \brief Gives a MetaIO according to its public name
      *
      * \param slt a public name of an output
      * \return pointer to the MetaIO referred by public name slt
      * \sa addSlot(MetaIO* slt)
      */
     MetaIO* getSlot(QString slt) { return sltlist[slt]; }

     /*! \brief gets the complete list of meta-signals (MetaIO) 
      * \return a dictionnary iterator over meta-signals
      */
     static QDictIterator<MetaIO> getSignalList() { return QDictIterator<MetaIO>(siglist); }
     /*! \brief gets the complete list of meta-slots (MetaIO) 
      * \return a dictionnary iterator over meta-slots
      */
     static QDictIterator<MetaIO> getSlotList() { return QDictIterator<MetaIO>(sltlist); }

     /*! \brief Retrieves the list of meta-slots related to a macro-block.
      * \param s name of the macro-block.
      * \return a list of slots names.
      */
     static QStrList getObjectSlots(QString s);
     /*! \brief Retrieves the list of meta-signals related to a macro-block.
      * \param s name of the macro-block.
      * \return a list of signals names.
      */
     static QStrList getObjectSignals(QString s);

private:
     QStringList objects;
     static QDict<MetaIO> siglist;
     static QDict<MetaIO> sltlist;
     QString filePath;
     MetaSheet* sheet;
     MetaBlock* block;
     bool instanciated;
     bool fileRepresentation;
};



#endif //__METABASE_H__
