/*
  name: statecanvas.h

  This file is part of ARCS - Augmented Reality Component System,
  written by Jean-Yves Didier, Vincent Le Ligeour and Yoann Petit
  for IBISC Laboratory (http://www.ibisc.univ-evry.fr)

  Copyright (C) 2004 Universit d'Evry-Val d'Essonne

  This program is free software: you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program.  If not, see <http://www.gnu.org/licenses/>.


  Please send bugreports  with examples or suggestions to
  didier__at__iup.univ-evry.fr
*/


#ifndef __STATECANVAS_H__
#define __STATECANVAS_H__

/*! \file transitioncanvas.h
 * \author Yoann Petit
 * \date 4 Juillet 2005
 * 
 * File describing transition canvas class
 */

#include <qcanvas.h>
#include <qdict.h>
#include "sheetview.h"
#include "transitioncanvas.h"
#include <metalibrary/metaconfigwriter.h>
#include "rtti.h"

/*! \brief StateCanvas CanvasItem description
 * 
 * This is the graphical counterpart of state included in state machines
 * \sa MetaStateMachine
 */
class StateCanvas : public QCanvasEllipse
{
public:

     /*! \brief StateCanvas constructor.
      *  \param *canvas Canvas of the StateMachineView.
      *  \param ns Name of the StateCanvas.
      */
     StateCanvas( QCanvas* canvas, QString ns);
     ~StateCanvas();

     /*! \brief Overload function of moveBy.
      *  Moving of the triangle and the TransitionCanvas.
      *  \param dx StateCanvas's moving along the X-axis.
      *  \param dy StateCanvas's moving along the Y-axis.
      */
     void moveBy(double dx, double dy);


     /*! \brief Returns the RTTI of the StateCanvas.
      *  \return RTTI of the StateCanvas.
      */
     int rtti() const { return RTTI_STATECANVAS; }

     /*! \brief This function is use to register a transition.
      *  \param *tc Pointer to a TransitionCanvas.
      */
     void registerTransition(TransitionCanvas *tc);

     /*! \brief Returns the name of the StateCanvas.
      *  \return Name of the StateCanvas.
      */
     QString getName() { return scName; }

     /*! \brief Returns a TransitionCanvas.
      *  \param id Id of the TransitionCanvas.
      *  \return TransitionCanvas.
      */
     TransitionCanvas* getTransition(QString id) { return dTransitionSender[id];}

     /*! \brief Rename the StateCanvas.
      *  \param newID New ID of the StateCanvas.
      */
     void renameStateID(QString newID);

     /*! \brief Add the TransitionCanvas to the XML tree.
      *  \param *exp Pointer to object that builds the XML tree.
      */
     void setTransition(MetaConfigWriter *exp); 
     
     /*! \brief Delete a TransitionCanvas of the StateCanvas.
      *  \param *tc The TransitionCanvas' sender.
      */
     void delTransitionSender(TransitionCanvas *tc) { dTransitionSender.remove(tc->getName()); }
     
     /*! \brief Delete a TransitionCanvas of the StateCanvas.
      *  \param *tc The TransitionCanvas' receiver.
      */
     void delTransitionReceiver(TransitionCanvas *tc);

     /*! \brief Returns an iterator to the dictionnary of TransitionCanvas "Sender".
      *  \return Iterator to the dictionnary of TransitionCanvas "Sender".
      */
     QDictIterator<TransitionCanvas> getTransitionIterator() { return QDictIterator<TransitionCanvas>(dTransitionSender); }

     /*! \brief Returns an iterator to the dictionnary of TransitionCanvas "Receiver".
      *  \return Iterator to the dictionnary of TransitionCanvas "Receiver".
      */
     QDictIterator<TransitionCanvas> getTransitionIteratorReceiver() { return QDictIterator<TransitionCanvas>(dTransitionReceiver); }

private:
     QDict<TransitionCanvas> dTransitionSender; //!< Dictionnary of all the TransitionCanvas that are sending.
     QDict<TransitionCanvas> dTransitionReceiver; //!< Dictionnary of all the TransitionCanvas that are receiving.
     QString scName; //!< Name of the StateCanvas.
     QCanvasText *textState; //!< Label for the name of the StateCanvas.
};


#endif //__STATECANVAS_H__
