/*
  name: rotator.h

  This file is part of ARCS - Augmented Reality Component System,
  written by Jean-Yves Didier, Vincent Le Ligeour and Yoann Petit
  for IBISC Laboratory (http://www.ibisc.univ-evry.fr)

  Copyright (C) 2004 Universit d'Evry-Val d'Essonne

  This program is free software: you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program.  If not, see <http://www.gnu.org/licenses/>.


  Please send bugreports  with examples or suggestions to
  didier__at__iup.univ-evry.fr
*/


#ifndef __ROTATOR_H__
#define __ROTATOR_H__

#include <qobject.h>
#include <cmath>
#include <qwaitcondition.h>
#include <iostream>

#ifndef M_PI
#define M_PI 3.14159265
#endif

/*! \brief This class is intended to be used with modified runtimes working without event loops.
 * 
 * The purpose of this class is to pause a runtime thread in case it is working without 
 * any event loop.
 */
class Sleeper : public QObject
{
Q_OBJECT

public:
     Sleeper(QObject* parent=0,const char* name=0) : QObject(parent, name) {} //!< ARCS Constructor

public slots:
     void start() { wc.wait(); } //!< Pausing the thread when started

private:
    QWaitCondition wc;
};


/*! \brief This class is a rotator object used to increment an angle.
 *
 * This class is a part of a sample. Its aim is to show how to integrate 
 * ARCS components and runtime to an application running with another
 * event loop.
 */
class Rotator : public QObject
{
Q_OBJECT

public:
     Rotator(QObject* obj=0, const char* name=0) : QObject(obj, name) //!< ARCS Constructor
    {
        angle = 0.0;
        step = 0.1;

    }

public slots:
     void setStep(float f) { step = f; } //!< Gives the increment in radian of the angle
     void invertStep() { step = -step; } //!< Transforms the increment in decrement.
     /*! \brief performs the actual increment.
      *
      * Triggers signal sendAngle().
      */
     void rotate()  
    { 
        if (angle >= M_PI)
            angle -= M_PI;
        if (angle <= -M_PI)
            angle += M_PI;
        angle += step ; 
        emit sendAngle(angle); 
    }

signals:
     /*! \brief Sends the actual angle.
      *
      * Triggered by rotate()
      */
        void sendAngle(float);

private:
    float angle;
    float step;

};

#endif //__ROTATOR_H__
