/*
  name: metaconfigreader.cpp

  This file is part of ARCS - Augmented Reality Component System,
  written by Jean-Yves Didier, Vincent Le Ligeour and Yoann Petit
  for IBISC Laboratory (http://www.ibisc.univ-evry.fr)

  Copyright (C) 2004 Universit d'Evry-Val d'Essonne

  This program is free software: you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program.  If not, see <http://www.gnu.org/licenses/>.


  Please send bugreports  with examples or suggestions to
  didier__at__iup.univ-evry.fr
*/


#include <metalibrary/metaconfigreader.h>
#include <iostream>
#include <qdir.h>
#include <qfile.h>
#include <qfileinfo.h>

using namespace std;

MetaConfigReader::MetaConfigReader()
{
     editMode = false;

}

MetaConfigReader::~MetaConfigReader()
{

}

bool MetaConfigReader::openFile(QString filename, QString p, bool per)
{
     QFile file( filename );
     if ( !file.open( IO_ReadOnly ) )
     {
	  cerr << "[META] couldn't open xml file." << endl;
	  return false;
     }
     if ( !application.setContent( &file ) ) 
     {
	  cerr << "[META] couldn't parse xml file." << endl;
	  file.close();
	  return false;
     }
    file.close();
    configpath = (QFileInfo(file)).dirPath();
    
    
    

    nameSpace = QString();
    metaBlock = false;

    if (! p.isEmpty())
    {
	 metaBlock = true;
	 //if (editMode)
	//	 nameSpace = filename + "::" ;
	 //else
	 	nameSpace = p + "::";
    }
    persistent = per;
    //if (editMode)
//	    prefix = filename;
  //  else
    	prefix = p;

    return true;
}


QDomElement  MetaConfigReader::findSection(QString s)
{
     QDomElement de = application.documentElement();

     if (!de.hasChildNodes())
     {
	  cerr << "[META] <application> markup without children." << endl;
	  return QDomElement();
     }

     QDomNodeList dnl = de.elementsByTagName(s);
     if (dnl.count() != 1)
     {
	  cerr << "[META] document has not the right number of <" << s.ascii() << "> markup" << endl;
	  return QDomElement();
     }

     return  dnl.item(0).toElement();
}


bool MetaConfigReader::parseDefineSection()
{
     QDomElement defines= findSection("defines");

     if (defines.isNull())
	  return true;
     
     if (!defines.hasChildNodes())
	  return true;

     QDomNodeList dnl2 = defines.elementsByTagName("define");
     for (unsigned int i=0; i < dnl2.count(); i++)
     {
	  QString s1 = dnl2.item(i).toElement().attribute("id");
	  QString s2 = dnl2.item(i).toElement().attribute("type");
	  QString s3 = dnl2.item(i).toElement().attribute("value");
	  if ( s2 == "object" )
	       s3 = nameSpace + s3 ;
	  new MetaDefine(nameSpace+s1,s2,s3);
     }
     return true;
}


bool MetaConfigReader::parseLibrariesSection()
{
     QDomElement libraries = findSection("libraries");

     if (libraries.isNull())
	  return false;

     if (!libraries.hasChildNodes())
	  return true;

     QDomNodeList dnl2 = libraries.elementsByTagName("library");
     for (unsigned int i = 0; i < dnl2.count(); i++)
     {
	  QString s = dnl2.item(i).toElement().attribute("name");
	  if (s == QString::null)
	       cerr << "[META] <library> markup without \"name\" attribute." << endl;
	  else
	  {
	       if ( QDir(s).isRelative() )
		    libraryNames.append( configpath + "/" + s);
	       else
		    libraryNames.append(s);

	  }
     } 
 
     return true;
}


bool MetaConfigReader::parseSignalsSection()
{
     QDomElement siglist = findSection("signals");

     if (siglist.isNull())
	  return true;

     if (!siglist.hasChildNodes())
	  return true;

     QDomNodeList dnl2 = siglist.elementsByTagName("method");
     for (unsigned int i = 0; i < dnl2.count(); i++)
     {
	  QString s1 = dnl2.item(i).toElement().attribute("name");
	  QString s2 = dnl2.item(i).toElement().attribute("object");
	  QString s3 = dnl2.item(i).toElement().attribute("method");

	  if ((s1 == QString::null) || (s2 == QString::null) || (s3 == QString::null))
	       cerr << "[META] <method> markup with wrong attributes." << endl;
	  else
	  {
	       MetaBlock* test = (MetaBlock*)MetaObject::getObject(prefix);
	       test->addSignal(new MetaIO(nameSpace + s1 , nameSpace + s2 , s3) );
//	       cout << "Registering " << (nameSpace + s1).ascii() << " signal " << endl;
	  }
     }
     return true;
}

bool MetaConfigReader::parseSlotsSection()
{
     QDomElement sltlist = findSection("slots");

     if (sltlist.isNull())
	  return true;

     if (!sltlist.hasChildNodes())
	  return true;

     QDomNodeList dnl2 = sltlist.elementsByTagName("method");
     for (unsigned int i = 0; i < dnl2.count(); i++)
     {
	  QString s1 = dnl2.item(i).toElement().attribute("name");
	  QString s2 = dnl2.item(i).toElement().attribute("object");
	  QString s3 = dnl2.item(i).toElement().attribute("method");
	  if ((s1 == QString::null) || (s2 == QString::null) || (s3 == QString::null))
	       cerr << "[META] <method> markup with wrong attributes." << endl;
	  else
	  {
	       MetaBlock* test = (MetaBlock*)MetaObject::getObject(prefix);
	       test->addSlot(new MetaIO(nameSpace+s1, nameSpace +s2,s3));
//	       cout << "Registering " << (nameSpace + s1).ascii() << " slot " << endl;
	  }
     }
     return true;

}



void MetaConfigReader::parseObjectElement(QDomElement e)
{
     QString s2 = e.attribute("id");
     bool p = true;

//     cout << "metablock value " << metaBlock << endl;

     if (metaBlock)
	  p = persistent ;
     else
     {
	  QString s3 = e.attribute("persistent");
	  if (s3 != QString::null) 
	       p = (s3.lower() == "true");
     }


     if ((s2 != QString::null) && metaBlock)
     {
//	  cout << " adding object " << (nameSpace + s2).ascii() << " to " << prefix.ascii() << endl;
	  MetaBlock* mb = (MetaBlock*)MetaObject::getObject(prefix);
	  if (mb == NULL)
	  {
	       cerr << "[META] Troubles fetching object " << prefix.ascii() << endl;
	       return ;
	  }
	  mb->addObject(nameSpace + s2);
     }



     if (! e.hasAttribute("file"))
     {
	  QString s1 = e.attribute("classname");
	  
	  if ((s1 == QString::null) || (s2 == QString::null))
	       cerr << "[META] <object> markup with wrong attributes." << endl;
	  else
	       new MetaObject(s1, nameSpace + s2,p);
	  return;
     }
     
     QString s1 = e.attribute("file");
     if ((s1 == QString::null) || (s2 == QString::null))
     {
	  cerr << "[META] <object> markup with wrong attributes." << endl;
	  return;
     }
    

     MetaConfigReader mr2;
     mr2.setEditMode(getEditMode());
     QString radix = s2;

     
     MetaBlock* mb = new MetaBlock(nameSpace +  s2,p);
     if (editMode)
     {
	     radix = s1;
	     MetaObject* mb2 = MetaObject::getObject(nameSpace + radix); 
	     if ( mb2 != NULL)
	     {
		     if (mb2->isBlock())
		     {
			     mb->linkToMetaBlock((MetaBlock*)mb2);
			     cerr << "[META] has found a block already described" <<endl;
			     return;
			}
		     cerr << "[META] Seems " << nameSpace.ascii() << radix.ascii() << " is not a metablock." << endl; 
	     }
	     else
	     {
	     	mb->linkToMetaBlock(new MetaBlock(nameSpace + radix, p));
	     	mb = mb->getMetaBlockLink();
	     }
     }

     
     /* Put here code to retrieve the real file */
     if ((QDir(s1)).isRelative())
     {
	  s1 = configpath + QDir::separator() + s1 ;
     }


     mb->setFile( s1 );

     //cout << "[META] parsing object " << s2.ascii() << endl ; //<< "[META] (prefix value : " << prefix.ascii() << endl;
     //cout << "[META] file " << s1.ascii() << endl;

     if ( ! mr2.openFile(s1,nameSpace+ radix,p))
	  cerr << "[Meta] unable to parse file " << s1.ascii() << endl;
     mr2.parseLibrariesSection();
     QStringList libs = mr2.getLibraryNames();
     for (unsigned int i=0; i < libs.count(); i++)
	  libraryNames.append(libs[i]);

     mr2.parseObjectsSection();
     mr2.parseSignalsSection();
     mr2.parseSlotsSection();
     mr2.parseDefineSection();
     // reste  lire la "feuille"
     //cout << "parsing sheet" << endl;
     MetaSheet* ms = mr2.parseSheetSection(radix);
     if (ms == NULL)
     {
	  cerr << "[META] A sheet is NULL, whereas it shouldn't." << endl;
	  return;
     }
      ms->setLinkedToBlock();
      mb->setMetaSheet(ms);
     //cout << "end of parsing sheet" << endl;
}



bool MetaConfigReader::parseObjectsSection()
{
     QDomElement objects = findSection("objects");

     if (objects.isNull())
	  return false;

     if (!objects.hasChildNodes())
	  return true;

     QDomNodeList dnl = objects.elementsByTagName("object");
     for (unsigned int i = 0; i < dnl.count(); i++)
     {	       
	  parseObjectElement(dnl.item(i).toElement());

     }
     return true;
}


MetaSheet* MetaConfigReader::parseSheetSection( QString s2)
{
    QDomElement sheet= findSection("sheet");
     if (sheet.isNull())
	  return NULL;
     //if (!sheet.hasChildNodes())
     //  return NULL;

     MetaSheet* ms = new MetaSheet(s2);
     if (!addPreConnections(ms, sheet))
	  return NULL;
     if (!addConnections(ms, sheet))
	  return NULL;
     if (!addPostConnections(ms, sheet))
	  return NULL;

     return ms;
}


bool MetaConfigReader::parseSheetsSection()
{
     QDomElement sheets = findSection("sheets");

     if (sheets.isNull())
	  return true;
     
     if (!sheets.hasChildNodes())
	  return true;

     parseDefineSection();

     QDomNodeList dnl = sheets.elementsByTagName("sheet");
     for (unsigned int i = 0; i < dnl.count() ; i++)
     {
	  QString s = dnl.item(i).toElement().attribute("id");
	  if (s == QString::null)
	       cerr << "[META] <sheet> markup without \"id\" attribute." << endl;
	  else
	  {
	       MetaSheet* ms = new MetaSheet(s);
	       if (!addPreConnections(ms, dnl.item(i).toElement()))
	       {
		    MetaDefine::clear();
		    return false;
	       }
	       if (!addConnections(ms, dnl.item(i).toElement()))
	       {
		    MetaDefine::clear();
		    return false;
	       }
	       if (!addPostConnections(ms, dnl.item(i).toElement()))
	       {
		    MetaDefine::clear();
		    return false;
	       }
	       if (!grepTokenSender(ms, dnl.item(i).toElement()))
	       {
		    MetaDefine::clear();
		    return false;
	       }
	  }
     }
     MetaDefine::clear();
     return true;
}


void MetaConfigReader::getSignalCouple(QString &a, QString &b, bool mode)
{
//     cout << "SigStep : " << a.ascii() << ", " << b.ascii() << endl;
     if (mode) 
	  return;

     MetaObject* obj = MetaObject::getObject(a);

     if (obj == NULL)
	  return;

     if (!obj->isBlock())
	  return;

     MetaIO* mio = ((MetaBlock*)obj)->getSignal(a + "::" + b);
     if (mio == NULL)
     {
//	  cout << "tried to grab signal " <<  (a+"::"+b).ascii() << endl;
	  return;
     }

     a = mio->getObjectName();
     b = mio->getIOObjectName();
     getSignalCouple(a,b,mode);
}

void MetaConfigReader::getSlotCouple(QString &a, QString &b, bool mode)
{
//     cout << "SltStep : " << a.ascii() << ", " << b.ascii() << endl;
     if (mode)
	  return;

     MetaObject* obj = MetaObject::getObject(a);

     if (obj == NULL)
	  return;

     if (!obj->isBlock())
	  return;

     MetaIO* mio = ((MetaBlock*)obj)->getSlot(a + "::" + b);
     a = mio->getObjectName();
     b = mio->getIOObjectName();
     getSlotCouple(a,b,mode);
}



void MetaConfigReader::addInitialisation(MetaSheet* ms, QDomNodeList lst, bool pre)
{
     //QDomNodeList dnl2 = dnl.item(0).toElement().elementsByTagName("init");
     for (unsigned int i =0; i < lst.count() ; i++)
     {
	  QString s1 = lst.item(i).toElement().attribute("object");
	  QString s4 = lst.item(i).toElement().attribute("slot");
	  QString s2 = lst.item(i).toElement().attribute("type");
	  QString s3 = lst.item(i).toElement().attribute("value");
	  if ( (s1 == QString::null) || (s2 == QString::null) || 
	       (s3 == QString::null) || (s4 == QString::null) ) 
	       cerr << "[META] <init> markup with wrong attributes." << endl;
	  else
	  {
	       s1 = nameSpace + s1;
	       getSlotCouple(s1,s4,editMode);
//	       cout << "init from " << s1.ascii() << " with slot " << s4.ascii() << endl;

	       if (s2 != "define")
	       {
		    if (s2 == "path")
		    {
			 if ((QDir(s3)).isRelative())
			 {
			      s3 = configpath + QDir::separator() + s3 ;
			 }
		    }

		    if (s2 == "object")
			 s3 = nameSpace + s3;

		    if (pre) 
			 ms->addPreConnection(MetaInit(s1, s4, s2, s3));
		    else
			 ms->addPostConnection(MetaInit(s1, s4, s2, s3));
	       }
	       else
		    if (pre)
			 ms->addPreConnection(MetaInit(s1, s4, s2, nameSpace+s3));
		    else
			 ms->addPostConnection(MetaInit(s1, s4, s2, nameSpace+s3));

	  }
		    
     }



}


bool MetaConfigReader::addPreConnections(MetaSheet* ms, QDomElement de)
{
     if (!de.hasChildNodes())
	  return true;

     QDomNodeList dnl = de.elementsByTagName("preconnection");
     if (dnl.count() > 1)
     {
	  cerr << "[META] Wrong number of <preconnection> markups in sheet " 
	       << ms->getSheetID().ascii() << " (" << dnl.count() << ")" << endl;
	  return false;
     }
     if (dnl.count() == 0) 
	  return true;
     

     addInitialisation(ms, dnl.item(0).toElement().elementsByTagName("init"),true);

     return true;
}


bool MetaConfigReader::addConnections(MetaSheet* ms, QDomElement de)
{
     if (!de.hasChildNodes())
	  return true;

     QDomNodeList dnl = de.elementsByTagName("connection");
     if (dnl.count() > 1)
     {
	  cerr << "[META] Wrong number of <connection> markups in one sheet" << endl;
	  return false;
     }
     if (dnl.count() == 0) 
	  return true;
     
     QDomNodeList dnl2 = dnl.item(0).toElement().elementsByTagName("wire");
     for (unsigned int i =0; i < dnl2.count() ; i++)
     {
	  QString s1 = dnl2.item(i).toElement().attribute("objsource");
	  QString s2 = dnl2.item(i).toElement().attribute("signal");
	  QString s3 = dnl2.item(i).toElement().attribute("objdest");
	  QString s4 = dnl2.item(i).toElement().attribute("slot");
	  if ( (s1 == QString::null) || (s2 == QString::null) ||
	       (s3 == QString::null) || (s4 == QString::null) )
	       cerr << "[META] <wire> markup with wrong attributes." << endl;
	  else
	  {
	       s1 = nameSpace + s1;
	       getSignalCouple(s1,s2,editMode); 
	       s3 = nameSpace + s3;
	       getSlotCouple(s3,s4,editMode);
	       //cout << "Registering connection " << s1.ascii() << ", " << s2.ascii() << " -> " << s3.ascii() << ", " << s4.ascii() << endl;
	       ms->addConnection(MetaWire(s1, s2, s3, s4));
	  }
     }

     return true;
}

bool MetaConfigReader::addPostConnections(MetaSheet* ms, QDomElement de)
{
     if (!de.hasChildNodes())
	  return true;

     QDomNodeList dnl = de.elementsByTagName("postconnection");
     if (dnl.count() > 1)
     {
	  cerr << "[META] Wrong number of <postconnection> markups in one sheet" << endl;
	  return false;
     }
     if (dnl.count() == 0) 
	  return true;
     
     addInitialisation(ms, dnl.item(0).toElement().elementsByTagName("init"), false);

     return true;
}


bool MetaConfigReader::configureStateMachine(MetaStateMachine* msm)
{
     QDomElement stateMachine = findSection("statemachine");

     if (stateMachine.isNull())
	  return false;


     QString s = stateMachine.attribute("terminal");
     if (s == QString::null)
	  msm->setTerminalState("end");
     else
	  msm->setTerminalState(s);

     if (stateMachine.hasAttribute("initial"))
     {
	  QString si = stateMachine.attribute("initial");
	  if (! si.isEmpty() )
	       msm->setFirstState(si);
     }

     if (!stateMachine.hasChildNodes())
	               return true;
     
     

     QDomNodeList dnl = stateMachine.elementsByTagName("transition");
     if (dnl.count() == 0)
	  cerr << "[META] Silly state machine found with no transition." << endl;


     for (unsigned int i=0; i < dnl.count() ; i++)
     {
	  QString s1 = dnl.item(i).toElement().attribute("stepA");
	  QString s2 = dnl.item(i).toElement().attribute("stepB");
	  QString s3 = dnl.item(i).toElement().attribute("token");

	  if (( s1 == QString::null) || (s2 == QString::null) || (s3 == QString::null))
	       cerr << "[META] <transition> markup with wrong attributes" << endl;
	  else
	       msm->insertTransition(s1, s2, s3);
     }
     return true;
}


bool MetaConfigReader::grepTokenSender(MetaSheet* ms, QDomElement de)
{
//     if (!de.hasChildNodes())
//	  return f;

     QDomNodeList dnl = de.elementsByTagName("tokensender");
     if (dnl.count() > 1)
     {
	  cerr << "[META] Wrong number of <tokensender> markups in one sheet" << endl;
	  return false;
     }
     if (dnl.count() == 0) 
	  return true;


     QString s = dnl.item(0).toElement().attribute("object");
     if ( s == QString::null)
	  return false;
     ms->setTokenSender(nameSpace + s );
//     cout << "Registering TokenSender " << (nameSpace + s).ascii() << endl;
     return true;
}
