/*
  name: metastatemachine.h

  This file is part of ARCS - Augmented Reality Component System,
  written by Jean-Yves Didier, Vincent Le Ligeour and Yoann Petit
  for IBISC Laboratory (http://www.ibisc.univ-evry.fr)

  Copyright (C) 2004 Universit d'Evry-Val d'Essonne

  This program is free software: you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program.  If not, see <http://www.gnu.org/licenses/>.


  Please send bugreports  with examples or suggestions to
  didier__at__iup.univ-evry.fr
*/


#ifndef __METASTATEMACHINE_H__
#define __METASTATEMACHINE_H__

/*! \file metastatemachine.h
 * \author Jean-Yves Didier
 * \date 07 Juin 2005
 * 
 * File describing routines associated to state machine.
 */

#include <qstring.h>
#include <qmap.h>
#include <qdict.h>
#include <iostream>
#include <metalibrary/metawin.h>

//using namespace std;

// For Visual C++ 6 compatibility
#ifdef WIN32
//template class DLL_EXPORT QMap<QString, QString>;
//template class DLL_EXPORT QDict<QMap<QString,QString>>;
#endif //WIN32

//! Class defining a finite state machine
/*!
 * This class defines a small mecanism to make efficiently state machines.
 * Developper just have to create the state machine, then declare all the
 * transitions he needs from one state to another state. At the end, a call 
 * to the start() method launches the state machine which begins at the first state 
 * given at the transition.
 */
class DLL_POINT MetaStateMachine
{
private:
     typedef QMap<QString,QString> MetaTokenAlternative;
     typedef QDict<MetaTokenAlternative> MetaTransition;

public:
     //! Constructor
     MetaStateMachine();
     ~MetaStateMachine();
     //! Flushes all the transitions in the state machine.
     void clear();
     /*! \brief Insert a transition.
      * 
      * Basically a state machine can be described as a list of states linked
      * together by transitions.
      + A transition is describing the condition to go from one state to another.
      * \param a First state
      * \param b Second state
      * \param t Token used to transit from first state to second state.
      */
     void insertTransition(QString a, QString b, QString t);
     //! Starts the state machine
     void start();
     /*! \brief Sets the initial (first) state of the state machine 
      * 
      * \param s Name of the initial state.
      */
     void setFirstState(QString s) { firstState = s; }
     /*! \brief Queries if the current state of the state machine is a terminal state.
      * 
      * What we call a terminal state is a state where the state machine as finished
      * its work.
      * \return true is teh current state is a terminal state, false otherwise.
      */
     bool isTerminalState() { return ((terminalState == state)&&(!terminalState.isEmpty())) ;}
     /*! \brief Sets the name of the terminal state
      * \param s name of the terminal state
      */
     void setTerminalState(QString s) { terminalState = s; } 
     /*! \brief Gives the name of the terminal state
      * \return the name of the terminal state
      */
     QString getTerminalState() { return terminalState; }
     /*! \brief Activates a transition
      * \param s name of the tpken used for transition
      * \return true if teh transition has been achieved, false otherwise
      */
     bool changeState(QString s);
     /*! \brief Return the current state of the state machine.
      * \return The current state of the state machine.
      */
     QString getCurrentState() {return state;}
     /*! \brief Gives the list of transitions starting from a given state
      * \param s name of the starting state
      * \return NULL if state doesn't exist, a QMap of (token, final state) pairs otherwise.
      */
     MetaTokenAlternative* getTransitionsStartingFrom(QString s) { return transitions.find(s); }

//private:
     DLL_POINT friend std::ostream& operator<<(std::ostream& os, const MetaStateMachine& msm);

     
private:
     QString state;
     QString terminalState;
     QString firstState;
     MetaTransition transitions;
};


typedef QMap<QString,QString> MetaTokenAlternative;
typedef QDict<MetaTokenAlternative> MetaTransition;


#endif //__METASTATEMACHINE_H__ 
