/*
  name: metaruntime.h

  This file is part of ARCS - Augmented Reality Component System,
  written by Jean-Yves Didier, Vincent Le Ligeour and Yoann Petit
  for IBISC Laboratory (http://www.ibisc.univ-evry.fr)

  Copyright (C) 2004 Universit d'Evry-Val d'Essonne

  This program is free software: you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program.  If not, see <http://www.gnu.org/licenses/>.


  Please send bugreports  with examples or suggestions to
  didier__at__iup.univ-evry.fr
*/


#ifndef __METARUNTIME_H__
#define __METARUNTIME_H__

/*! \file metaruntime.h  
 * File providing the runtime routines.
 * \author Jean-Yves Didier
 * \date 07 Juin 2005
 */

#include <qobject.h>
#include <qapplication.h>
#include <metalibrary/metabase.h>
#include <metalibrary/metalibrary.h>
#include <metalibrary/metastatemachine.h>
#include <metalibrary/metaconfigreader.h>

#include <metalibrary/metawin.h>


/*! \todo Runtime classes should be moved to libarcs.
 * This way, the use of the runtime as a derived class would be more 
 * approriate. Any derivative work would be more groovy.
 */ 

/*! Class providing a runtime for an application.
 *
 */
class DLL_POINT MetaRuntime : public QObject
{
Q_OBJECT

public:
/*! \brief constructor
 * \param parent provided for compatibility for QT
 * \param name provided for compatibility for QT
 */
MetaRuntime(QObject* parent=0, const char* name="runtime");
~MetaRuntime();

/*! \brief Loads the configuration file.
 *
 * A configuration file has the structure described in MetaConfigReader
 * \param filename Name of the config file.
 * \return true if config file loading was successfull, false otherwise.
 */
bool loadConfigFile(QString filename);
//! \brief Loads all the libraries needed in the config file.
bool loadLibraries(bool b=true);
//! \brief Instanciate all the objects needed in the config file.
bool instanciateObjects();
//! \brief Instanciate all the sheets needed in the config file.
bool instanciateSheets();
//! \brief Instanciate the state machine.
bool instanciateStateMachine();

//! \brief For compatibility purpose.
bool event(QEvent* e);

/*! \brief instanciate an object according to its description.
 *
 * The description of such an object is put in class MetaObject.
 * \param mobj A pointer to the MetaObject descriptor.
 * \return <tt>true</tt> if succeeded, <tt>false</tt> otherwise.
 */
bool instanciateObject(MetaObject* mobj);

/*! \brief destroy an object according to its description.
 *
 * The description of such an object is put in class MetaObject.
 * \param mobj A pointer to the MetaObject descriptor.
 * \return <tt>true</tt> if succeeded, <tt>false</tt> otherwise.
 */
bool destroyObject(MetaObject* mobj);


void setEventDriven(bool b) { eventDriven = b; }
bool isEventDriven() { return eventDriven; }

bool hasFinished() { return stopped; }

/*! \brief activate the next sheet
 */
void nextSheet() { activateSheet(msm.getCurrentState()); }

//! \brief Dumps all the content of basic types instanciated so far.
void dump();

/*!\brief starts the runtime by parsing a configuration file.  
 *
 * This method is equivalent to the old way of starting the runtime (see main.cpp in this directory).
 * \param fileName name of the file to load with this runtime
 * \param evtDriven tells wether this runtime should be event driven or not.
 * \return error codes : <ul>
 * <li>0: everything is ok !</li>
 * <li>2: config file has not been loaded</li>  
 * <li>3: libraries fas not been loaded properly</li>
 * <li>4: objects have not been instanciated properly</li>
 * <li>5: sheets have not been created properly</li>
 * <li>6: state machine have not been set up properly</li>
 * </ul>
 */
int startRuntime(QString fileName, bool evtDriven = true);


public slots:
//     void switchSheet();
//! \brief Slot to recieve the right token.
void recieveToken(QString token);
//! \brief starts the runtime core routines.
void run();



signals:
void finished();//!< signal emitted once the terminal state of the statemachine is reached.
void init(); //!< signal to init empty initializations
void init(bool);//!< signal to init boolean initializations
void init(int);//!< signal to init integer initializations
void init(double);//!< signal to init double initializations
void init(float);//!< signal to init float initializations
void init(QString);//!< signal to init QString initializations
void init(QObject*);//!< signal to init QObject* initializations

private:
void cleanup(); //!< needed so that arcs applications really quit.
void activateSheet(MetaSheet* ms);
void activateSheet(QString sheetName) { if (!sheetName.isEmpty()) activateSheet(MetaSheet::getSheet(sheetName));}
void deactivateSheet(MetaSheet* ms);     
void deactivateSheet(QString sheetName) { if (!sheetName.isEmpty()) deactivateSheet(MetaSheet::getSheet(sheetName)); }
void initialize(MetaInit mi);//QString objectID, QString slot, int type, QString value);
void connect(MetaWire mw);
void disconnect(MetaWire mw);

MetaLibrary ml;
MetaConfigReader mcr;
MetaStateMachine msm;
bool eventDriven;

bool stopped;
bool sheetCurrentlyActivated;

QString configFile;

};


#endif 
