/*
  name: metalibrarytoolkit.h

  This file is part of ARCS - Augmented Reality Component System,
  written by Jean-Yves Didier, Vincent Le Ligeour and Yoann Petit
  for IBISC Laboratory (http://www.ibisc.univ-evry.fr)

  Copyright (C) 2004 Universit d'Evry-Val d'Essonne

  This program is free software: you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program.  If not, see <http://www.gnu.org/licenses/>.


  Please send bugreports  with examples or suggestions to
  didier__at__iup.univ-evry.fr
*/


#ifndef __METALIBRARYTOOLKIT_H__
#define __METALIBRARYTOOLKIT_H__

//#define DEBUG

/*! \file metalibrarytoolkit.h
 *  \author Jean-Yves Didier 
 *  \date October, 15th 2004
 * 
 * This the header declaring basic classes and functionnalities 
 * to easily create a library designed for our rutime. It stores 
 * a generic mecanism of wrapping that helps to call the constructor of any
 * object inheriting from QObject.
 * One can create it's own library for our runtime, by including this file in
 * the source code of its own library.
 * 
 * \par 
 * <B>How to create its own library ?</B>
 *
 * First, the developper has to create all the objectshe wants to include in the library.
 * Then in these objects, one can want to make some of them accessible through the 
 * dynamic library.
 * We recommend then to create a library file source, that will of course include
 * all the headers of the previously defined classes and then include this header (metalibrarytoolkit).
 * By using the macros we implemented anyone will be able to declare the objects (inheriting from <TT>QObject</TT>
 * he wants to export. This file should look like this  :
 * 
 * \code
 * #include <metalibrarytoolkit.h>  // for macros definitions 
 * #include "class1.h"  // Header supposed to contain a user class "Class1" definition
 * #include "class2.h"  // Same as above for "Class2"
 * 
 * METALIB_BEGIN
 * METALIB_OBJECT(Class1)
 * METALIB_OBJECT(Class2)
 * METALIB_END
 * \endcode
 * 
 * Then, if it is compiled as a dynamic library, two entry point will be created :
 * \code
 * void metalib_fill(MetaMap map);
 * MetaMap metalib_map();
 * \endcode
 *
 * The first one is completing a already existing MetaMap with objects references that will be
 * completed with internal references of the library.
 * The second is returning the own's library MetaMap.
 */

//#include <qmap.h>
#include <qdict.h>
#include <qobject.h>
#include <qstring.h>
#include <iostream>
#include <qaction.h>

//! Generic class to wrap all the constructors of objets stored in libraries
/*! This class has a generic behaviour and is acting as an object factory.
 */
class MetaWrapper
{
public:
     /*! \brief Wrapper to call a constructor of any objects heriting from QObject.
      * A wrapper to call the constructor of any objects heriting from QObject
      */
     virtual QObject* createObject(QObject* obj, const char* name) {return new QObject(obj,name); }
     virtual void destroyObject(QObject* obj) {};
};


//! Template class that inherits from MetaWrapper.
/*! Basically it shares the same method and allows to call the right constructor of the objects we try to create.
 */
template<class X> class MetaWrapperTemplate : public MetaWrapper
{
public:
     //! Same method as in MetaWrapper, only the implementation differs.
     QObject* createObject(QObject* obj, const char* name);
     //! Same method as in MetaWrapper, only the implemantation differs.
     void destroyObject(QObject* obj);
};


#ifndef DLL_EXPORT
#ifdef WIN32
/*! Macro defining keywords for dllexports for MSVC
 *
 * This macro will help export necessary dll entry points in Microsoft Windows environment
 */
#define DLL_EXPORT __declspec(dllexport) 
template class DLL_EXPORT QDict<MetaWrapper>;
#else
#define DLL_EXPORT
#endif //WIN32
#endif //DLL_EXPORT


/*! Type storing wrapping classes in one end and a key, which is a string representing
 * the name of the object we want create.
 */
typedef QDict<MetaWrapper> MetaMap;


/*! Dirty way to implement wrapper since it is in the header file (don't do this at home !), 
 * basically for the sole purpose to only have one file 
 * to include with the user library.
 */
template<class X> 
QObject* MetaWrapperTemplate<X>::createObject(QObject* parent, const char* name)
{
     return (new X(parent,name));
}

//! Dirty (Kids, don't do this at home !)
template<class X>
void MetaWrapperTemplate<X>::destroyObject(QObject* obj)
{
     delete ( X *)obj;
}


/*! Macro acting as a markup signaling the beginning of a user library designed for our system that one wants to implement.
 *
 * Basically, this macro is starting the implementation of a particular "C" function called "metalib_fill"
 * which can be used as an entrance point in the library.
 */
#define METALIB_BEGIN \
extern "C" { \
DLL_EXPORT void metalib_fill(MetaMap* map) \
{ 


/*! Macro helping to associate an object constructor with a string representing its name
 */
#define METALIB_OBJECT(A) (*map).insert(#A, new MetaWrapperTemplate<A>());

/*! Macro acting as a markup signaling the end of a user library
 * 
 * It's also implementing a "C" function called "metalib_map" 
 * returning a MetaMap of strings associated to objects's constructors.
 */
#define METALIB_END \
} \
DLL_EXPORT MetaMap metalib_map() { MetaMap map; metalib_fill(&map); return map;} \
}


#endif //__METALIBRARYTOOLKIT_H__
