/*
  name: metalibrary.h

  This file is part of ARCS - Augmented Reality Component System,
  written by Jean-Yves Didier, Vincent Le Ligeour and Yoann Petit
  for IBISC Laboratory (http://www.ibisc.univ-evry.fr)

  Copyright (C) 2004 Universit d'Evry-Val d'Essonne

  This program is free software: you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program.  If not, see <http://www.gnu.org/licenses/>.


  Please send bugreports  with examples or suggestions to
  didier__at__iup.univ-evry.fr
*/


#ifndef __METALIBRARY_H__
#define __METALIBRARY_H__

/*! \file metalibrary.h
 * \author Jean-Yves Didier
 * \date 07 Juin 2005
 * 
 * A file containing the definition of MetaLibrary class, which manages user libraries
 */

#include <metalibrary/metalibrarytoolkit.h>
#include <metalibrary/metawin.h>
#include <qobject.h>
#include <qmap.h>
#include <qstring.h>
#include <qstringlist.h>
#include <qlibrary.h>
#include <qdict.h>
#include <qstrlist.h>

/*! Class managing user libraries 
 * 
 * This class provides mecanism to manage user libraries : 
 * load libraries, query objects declared in it, instanciate objects
 * contained in library, query signals
 * that an object can broadcast, query slots that an object has.
 *
 */
class DLL_POINT MetaLibrary
{
public:
     //! Constructor
     MetaLibrary();
	 MetaLibrary(const MetaLibrary & lib);

	 //! Destructor
	 ~MetaLibrary();


     //!  Loads a dynamic user library
     /*!
      * \param s filename of the library
      * \return true if the library was successfully loaded, false otherwise.
7      */
     bool loadLibrary(QString s );
     //! Gets the global name list of the objects stored in all libraries loaded with this class.
     /*!
      * \return The global name list of the objects stored in all libraries loaded.
      */
     QStringList getObjects();
     //! Gets the name list of all libraries successfully loaded.
     /*!
      * \return The name list of all libraries successfully loaded.
      */
     QStringList getLibraries();
     //! Gets the name list of object signals
     /*!
      * This method query it for a class name. If you want to query it for an already created object,
      * you might prefer use the method  MetaLibrary::getObjectSignals(QObject* obj)
      * \param objName class Name of the object of which we want to query signals.
      * \param super if set to <tt>true</tt>, it will also list signals inherited by superclass
      * \return a list of signals names.
      */
     QStrList getObjectSignals(QString objName,bool super=false);
     //! Gets the name list of object signals
     /*!
      * This method query it for an already existing object. 
      * \param obj Pointer on a Qt Object of which we want to query signals.
      * \param super if set to <tt>true</tt>, it will also list signals inherited by superclass
      * \return a list of signals names.
      */
     QStrList getObjectSignals(QObject* obj,bool super=false);
     //! Gets the name list of object slots
     /*!
      * This method query it for a class name. If you want to query it for an already created object,
      * you might prefer use the method  MetaLibrary::getObjectSlots(QObject* obj)
      * \param objName class Name of the object of which we want to query slots.
      * \param super if set to <tt>true</tt>, it will also list signals inherited by superclass
      * \return a list of slots names.
      */
     QStrList getObjectSlots(QString objName, bool super=false);
     //! Gets the name list of object slots
     /*!
      * This method query it for an already existing object. 
      * \param obj  Pointer on a Qt Object of which we want to query slots.
      * \param super if set to <tt>true</tt>, it will also list slots inherited by superclass
      * \return a list of slots names.
      */
     QStrList getObjectSlots(QObject* obj, bool super=false);
     /*! \brief Instanciate an object using its class name.
      * \param objName Classname of the object.
      * \param parent kept for compatibility purpose (parent of the object).
      * \param name kept for compatibility purpose (name of the object).
      */
     QObject* createObject(QString objName,QObject* parent = 0, const char* name = 0);

     /*! \brief Destroy an object using its ID.
      * \param objName Classname of the object.
      * \param obj Pointer to the object.
      */
     void destroyObject(QObject* obj, QString objName); 

private:
	 //! Needed to convert unix lib names to windows names and so on
	 /*!
	  *	\param s library file name to convert (s should be a path).
	  * \return the converted file name
	  */
	 QString genericLibName(QString s);
     MetaMap objectMap;
     QDict<QLibrary> libraryMap; 

};


/*!
 * Type declaring for function's pointer on user's libraries entry point.
 * see documentation about file metalibrarytoolkit.h about these
 * entry points
 */
typedef void (*MetaLibraryFill) (MetaMap* map);


#endif //__METALIBRARY_H__
