/*
  name: metaconfigreader.h

  This file is part of ARCS - Augmented Reality Component System,
  written by Jean-Yves Didier, Vincent Le Ligeour and Yoann Petit
  for IBISC Laboratory (http://www.ibisc.univ-evry.fr)

  Copyright (C) 2004 Universit d'Evry-Val d'Essonne

  This program is free software: you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program.  If not, see <http://www.gnu.org/licenses/>.


  Please send bugreports  with examples or suggestions to
  didier__at__iup.univ-evry.fr
*/


#ifndef __METACONFIGREADER_H__
#define __METACONFIGREADER_H__

#include <qdom.h>
#include <qstringlist.h>
#include <metalibrary/metabase.h>
#include <metalibrary/metastatemachine.h>

/*!
 *  \todo Should test if it works without black magic now !
 */
//#undef DLL_POINT 
//#include <metalibrary/metawin.h>

/*! \file metaconfigreader.h
 * \author Jean-Yves Didier
 * \date 06 Juin 2005
 *
 * File gathering one class : MetaConfigReader which role is to parse the content 
 * of an XML description of an application.
 * This class allows to read the XMl description of both applications and macro-blocks.
 * As a matter of consequences, there are two xml descriptions to expose.
 * 
 * The DTD of the xml application description is as follows :
 * \include application.dtd
 * The DTD of the xml macro-block description is as follows :
 * \include macroblock.dtd
 */

/*! \brief Class used to parse and read content of the xml formats designed for this runtime.
 * 
 * This class allows to read the XML description of both applications and macro-blocks.
 * As a matter of consequences, there are two xml descriptions that are required to know. 
 *
 */

class DLL_POINT MetaConfigReader
{
public:
     MetaConfigReader();
     ~MetaConfigReader();

     /*! \brief Open an xml file describing an application.
      * 
      * \param filename Name of the XML file to load.
      * \param p Prefix to use for meta-blocks namespace
      * \param per Persistent attribute used with macro-blocks.
      * \return true if the loading was successfull, false otherwise.
      */
     bool openFile(QString filename, QString p = QString::null, bool per=true);
     /*! \brief Allows to get the name of libraries the user wants for his application.
      *
      * Method parseLibrariesSection() must be called before in order to obtain library names.
      * \return A list of library names.
      * \sa parseLibrariesSection()
      */
     QStringList getLibraryNames() { return libraryNames; }
     
     /*! \brief Parses the signals section.
      *
      * The signals section is typical of macro-blocks only.
      * Don't try to use it to parse an XML application description.
      * This method will create MetaIO objects.
      * \return true if the signals section was well parsed, false otherwise.
      */
     bool parseSignalsSection();

     /*! \brief Parses the slots section.
      *
      * The slots section is typical of macro-blocks only.
      * Don't try to use it to parse an XML application description.
      * This method will create MetaIO objects.
      * \return true if the slots section was well parsed, false otherwise.
      */
     bool parseSlotsSection();

     /*! \brief Gets a couple of (object ID, signal name).
      *
      * This method recovers the real couple (object ID, signal name) on which a connection is performed.
      * It is intended mostly for the macro-block use.
      * The initial couple being the macro-block ID and its public signal is transformed 
      * into an object ID internal to the macro-block and the corresponding signal.
      *
      * \param a The object ID of the couple
      * \param b The signal name of the couple
      * \return \a a and \a b are modified to return the real couple on which the connection  is made.
      */
     static void getSignalCouple(QString &a, QString &b, bool mode=false);

     /*! \brief Gets a couple of (object ID, silot name).
      *
      * This method recovers the real couple (object ID, signal name) on which a connection or an initialisation is performed.
      * It is intended mostly for the macro-block use.
      * The initial couple being the macro-block ID and its public slot is transformed 
      * into an object ID internal to the macro-block and the corresponding slot.
      *
      * \param a The object ID of the couple.
      * \param b The slot name of the couple.
      * \return \a a and \a b are modified to return the real couple on which the connection or the initialisation is made.
      */
     static void getSlotCouple(QString &a, QString &b, bool mode=false);


     /*! \brief Parses an Object Element
      *
      * This method purpose is to parse object section.
      * It will take the appropriate measures if the object is a pure object or a macro-block.
      * \param e The element of the XML tree that is related to an object.
      */
     void parseObjectElement(QDomElement e);

     /*! \brief Parses a sheet section.
      *
      * This method will create a MetaSheet object related to a sheet section of a document, either an application or a macro-block.
      * \param name name of the sheet section.
      * \return a pointer to the MetaSheet object created, <tt>NULL</tt> otherwise.
      */
     MetaSheet* parseSheetSection(QString name);

     /*! \brief sets the reader in edit mode or not.
      *
      * The edit mode is used by graphical editor.
      * \param mode A bool telling if in edit mode or not (not is considered as runtime mode).
      */
     void setEditMode(bool mode) { editMode = mode; }
     /*! \brief Tells wether we're in edit mode or not
      *
      * \return true if in edit mode, false otherwise.
      */
     bool getEditMode() { return editMode; }

     /*! \brief Parses the defines section.
      *
      * This method will create MetaDefine objects.
      * \return true if the defines section was well parsed, false otherwise.
      */
     bool parseDefineSection();
     /*! \brief Parses the libraries section.
      *
      * This method must be called before getLibaryNames() is invocated.
      * \sa getLibraryNames()
      * \return true if the library section was well parsed, false otherwise.
      */
     bool parseLibrariesSection();
     /*! \brief Parses the objects section
      *
      * This method will create MetaObject objects.
      * Those can be accessed using the MetaObject::getObject(QString s) method.
      * \return true if the objects section was well parsed, false otherwise.
      */
     bool parseObjectsSection();
     /*! \brief Parses the sheets section
      *
      * This method will create MetaSheet objects.
      * \return true if the sheets section was well parsed, false otherwise.
      */
     bool parseSheetsSection();
     /*! \brief Parses the statemachine section and configure the state machine
      *
      * This method will configure all the transitions needed by the state machine to perform its work.
      * \param msm The MetaStateMachine to configure.
      * \return true if the state machine could be configured, false otherwise.
      */
     bool configureStateMachine(MetaStateMachine* msm);

private:
     QDomElement  findSection(QString s);
     void addInitialisation(MetaSheet* ms, QDomNodeList lst, bool pre);
     bool addPreConnections(MetaSheet* ms,QDomElement de);
     bool addPostConnections(MetaSheet* ms, QDomElement de);
     bool addConnections(MetaSheet* ms, QDomElement de);
     bool grepTokenSender(MetaSheet* ms, QDomElement de);
     QString prefix;
     QString nameSpace;
     bool persistent;
     bool metaBlock;
     bool editMode;

     QString configpath;

     QDomDocument application;
     QStringList libraryNames;

};




#endif //__METACONFIGREADER_H__
