/*
  name: wirecanvas.h

  This file is part of ARCS - Augmented Reality Component System,
  written by Jean-Yves Didier, Vincent Le Ligeour and Yoann Petit
  for IBISC Laboratory (http://www.ibisc.univ-evry.fr)

  Copyright (C) 2004 Universit d'Evry-Val d'Essonne

  This program is free software: you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program.  If not, see <http://www.gnu.org/licenses/>.


  Please send bugreports  with examples or suggestions to
  didier__at__iup.univ-evry.fr
*/


#ifndef __WIRECANVAS_H__
#define __WIRECANVAS_H__

/*! \file wirecanvas.h
 * \author Yoann Petit
 * \date 1 Juillet 2005
 * 
 * File describing wirecanvas and nodeitem class
 */

#include <metalibrary/metabase.h>
#include "gripwire.h"
#include "linewire.h"
#include "rtti.h"

/*! \brief Node CanvasItem description
 * 
 * This is one of the graphical counterpart of ObjectCanvas included in an object
 * \sa ObjectCanvas
 */
class NodeItem : public QCanvasPolygon
{
public:

     /*! \brief NodeItem constructor.
      *  \param *oc ObjectCanvas of the NodeItem.
      *  \param x NodeItem position along the X-axis on the sheet.
      *  \param y NodeITem position along the Y-axis on the sheet.
      *  \param label Name of the signal or slot.
      *  \param isSlot Type of NodeItem, true for a slot, false for a signal.
      */
     NodeItem(QCanvasItem* oc, int x, int y, QString label, bool isSlot);
     ~NodeItem();

     /*! \brief Overload function of moveBy.
      *  Moving of the triangle and the label.
      *  \param x NodeItem's moving along the X-axis.
      *  \param y NodeItem's moving along the Y-axis.
      */
     void moveBy(double dx, double dy);

     /*! \brief Overload function of setZ.
      *  Define the depth of the NodeItem.
      *  \param z Depth of the NodeItem.
      */
     void setZ(double z);

     /*! \brief Returns the RTTI of the NodeItem.
      *  \return RTTI of the NodeItem.
      */
     int rtti() const { return RTTI_NODEITEM; }

     /*! \brief Returns the NodeItem's property.
      *  \return True if the NodeItem is a slot, false if is a signal.
      */
     bool isSlot() { return slotProperty;}

     /*! \brief Returns the width of the NodeItem.
      *  \return Width of the NodeItem.
      */
     int getWidth() {return width;}

     /*! \brief Returns the height of the NodeItem.
      *  \return Height of the NodeItem.
      */
     int getHeight() {return height; }

     /*! \brief Returns the name of the NodeItem.
      *  \return Name of the NodeItem.
      */
     QString getNodeName() { return nodeName; }

     /*! \brief Returns the ObjectCanvas of the NodeItem.
      *  \return QCanvasItem. The ObjetcCanvas of the NodeItem.
      */
     QCanvasItem* getObjectCanvas() { return ocNode; }

private:
     QCanvasText *sLabel; //<! Label for show the name of the NodeItem
     QString nodeName; //<! Name of the NodeItem
     QCanvasItem *ocNode; //!< ObjectCanvas of the NodeItem
     int height; //<! Height of the NodeItem
     int width; //<! Width of the NodeItem
     bool slotProperty; //<! Property of the NodeItem

};

/*! \brief Wire CanvasItem description
 * 
 * This is the graphical counterpart of connection included in a sheet
 */
class WireCanvas : public QCanvasLine
{
public:

     /*! \brief Wirecanvas constructor.
      *  \param *mw MetaWire of the WireCanvas.
      *  \param *from NodeItem of source.
      *  \param *to NodeItem of destination.
      *  \param p1 Position along the Y-axis of the object of source on the sheet.
      *  \param h1 Height of the Object of source.
      *  \param p2 Position along the Y-axis of the object of destination on the sheet.
      *  \param h2 Height of the Object of destination.
      */
     WireCanvas(MetaWire* mw, NodeItem *from, NodeItem *to,
			    int p1, int h1, int p2, int h2);
     ~WireCanvas();

     /*! \brief Returns the RTTI of the WireCanvas.
      *  \return RTTI of the WireCanvas.
      */
     int rtti() const { return RTTI_WIRECANVAS; }

     /*! \brief Sets points of GripWires and LineWires of the WireCanvas.\n
      *  setFromPoint call setPoint.\n
      *  setFromPoint is called when an ObjectCanvas' source of a WireCanvas moving and replace the first LineWire.
      *  \param x ObjectCanvas' moving along the X-axis.
      *  \param y ObjectCanvas' moving along the Y-axis.
      */
     void setFromPoint( int x, int y );

     /*! \brief Sets points of GripWires and LineWires of the WireCanvas.\n
      *  setToPoint call setPoint.\n
      *  setToPoint is called when an ObjectCanvas' destination of a WireCanvas moving and replace the last LineWire.
      *  \param x ObjectCanvas' moving along the X-axis.
      *  \param y ObjectCanvas' moving along the Y-axis.
      */
     void setToPoint( int x, int y );

     /*! \brief Sets points of GripWires and LineWires of the WireCanvas.
      *  \param x ObjectCanvas' moving along the X-axis.
      *  \param y ObjectCanvas' moving along the Y-axis.
      */
     void setPoint(int x, int y);


     void getToPoint(int& x, int &y) { x = lineWire[4]->endPoint().x(); y = lineWire[4]->endPoint().y(); }
     void getFromPoint(int& x, int& y) { x= lineWire[0]->startPoint().x(); y = lineWire[0]->startPoint().y();}



     /*! \brief set visible of GripWires of the WireCanvas.
      */
     void setVisibleGrip(bool visible);

     /*! \brief Returns the MetaWire of the WireCanvas.
      *  \return MetaWire of the WireCanvas.
      */
     MetaWire* getMetaWire() { return myWire; } 

     /*! \brief Returns the NodeItem's source of the WireCanvas.
      *  \return NodeItem's source  of the WireCanvas.
      */
     NodeItem* getFromNode() { return myFrom; }

     /*! \brief Returns the NodeITem's destination of the WireCanvas.
      *  \return NodeItem's destination of the WireCanvas.
      */
     NodeItem* getToNode() { return myTo; }

     /*! \brief Returns a GripWire of the WireCanvas.
      *  \param id Number of the GripWire.
      *  \return GripWire of the WireCanvas.
      */
     GripWire* getGripPoint(int id) { return gripPoint[id]; }

     /*! \brief Returns a LineWire of the WireCanvas.
      *  \param id Number of the LineWire.
      *  \return Pointer to a LineWire of the WireCanvas.
      */
     QCanvasLine* getLineWire(int id) { return lineWire[id]; }
     


 
private:
     MetaWire* myWire; //<! MetaWire of the WireCanvas
     NodeItem* myFrom; //<! NodeItem's source of the WireCanvas
     NodeItem* myTo; //<! NodeItem's destination of the WireCanvas
     LineWire *lineWire[5]; //!< Array of LineWires
     GripWire *gripPoint[4]; //<! Array of GripWires
     bool ThreeToFive; //<! Allows to know if the wire is passing to three at five lines
     bool FiveToThree; //<! Allows to know if the wire is passing to five at three lines

};

#endif //__WIRECANVAS_H__

