/*
  name: statemachineview.h

  This file is part of ARCS - Augmented Reality Component System,
  written by Jean-Yves Didier, Vincent Le Ligeour and Yoann Petit
  for IBISC Laboratory (http://www.ibisc.univ-evry.fr)

  Copyright (C) 2004 Universit d'Evry-Val d'Essonne

  This program is free software: you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program.  If not, see <http://www.gnu.org/licenses/>.


  Please send bugreports  with examples or suggestions to
  didier__at__iup.univ-evry.fr
*/


#ifndef __STATEMACHINEVIEW_H__
#define __STATEMACHINEVIEW_H__

/*! \file transitioncanvas.h
 * \author Yoann Petit
 * \date 4 Juillet 2005
 * 
 * File describing transition canvas class
 */

#include <qcanvas.h>
#include <qdict.h>
#include "objectcanvas.h"
#include <metalibrary/metabase.h>
#include "wirecanvas.h"
#include <metalibrary/metalibrary.h>
#include <qpoint.h>
//#include <qqueue.h>
#include <qevent.h>
#include <qstringlist.h>
#include "statecanvas.h"
#include "transitioncanvas.h"
#include <metalibrary/metaconfigwriter.h>

/*! \brief State Machine View description
 */
class StateMachineView : public QCanvasView
{
    Q_OBJECT
public:

    /*! \brief StateMachineView constructor.
     *  \param *parent Parent of the StateMachineView.
     *  \param *name Name of the StateMachineView.
     *  \param InitialID ID of the initial state of the StateMachineView.
     *  \param TerminalID ID of the initial state of the StateMachineView.
     */
    StateMachineView( QWidget *parent, const char *name, QString InitialID = 0, QString TerminalID = 0 );
    ~StateMachineView();

    /*! \brief Register a state.
      * \param *sc The StateCanvas of the State.
      */
    void registerState( StateCanvas *sc );

    /*! \brief Returns if a state is on the StateMachineView.
     *  \param id Name of the state.
     *  \return True if the state id is already on the StateMachineView.
     */
    bool isState(QString id);

    /*! \brief Add a transition on the state machine.
     *  \param *tc The transition to add.
     */
    void addTransition(TransitionCanvas *tc);
    
    /*! \brief Returns a StateCanvas.
     *  \param id Id of the state.
     *  \return StateCanvas with ID "id".
     */
    StateCanvas* getStateCanvas(QString id) { return dState[id]; }

    /*! \brief Returns a LineWire of the WireCanvas.the number of state on the state machine.
     *  \return Number of state on state machine.
     */
    int getNbrState() { return dState.count(); }
    
    /*! \brief Returns the ID of the terminal state of the state machine.
     *  \return ID of the terminal state.
     */
    QString getTerminalState() { return stateTerminalID; }

    /*! \brief Returns the ID of the initial state of the state machine.
     *  \return ID of the initial state.
     */
    QString getInitialState() { return stateInitialID; }

    /*! \brief Sets the terminal state of the state machine.
     *  \param id ID of the terminal state.
     */
    void setTerminalState(QString id);

    /*! \brief Sets the initial state of the state machine.
     *  \param id ID of the initial state.
     */
    void setInitialState(QString id);

    /*! \brief Renames a state.
     *  \param ID Old ID of the state.
     *  \param NewID New ID of the state.
     */
    void renameStateID(QString ID, QString newID);

    /*! \brief Deletes a state.
     *  \param ID ID of the state.
     */
    void delState(QString ID);

    /*! \brief Deletes all states and reinitalizes IDs of the initial and terminal state.
     */
    void clear();

    /*! \brief Add the Transition to the XML tree.
     *  \param *exp Pointer to object that builds the XML tree.
     */
    void setTransition(MetaConfigWriter *exp);   

    //QDict<StateCanvas> getState() {return dState;}

    /*! \brief Returns an iterator to the dictionnary of StateCanvas.
     *  \return Iterator to the dictionnary of StateCanvas.
     */
    QDictIterator<StateCanvas> getStateIterator() { return QDictIterator<StateCanvas>(dState);}

protected:

    /*! \brief When the user press a button of the mouse.
     */
    void contentsMousePressEvent( QMouseEvent* );

    /*! \brief When the user moves the mouse. 
     */
    void contentsMouseMoveEvent( QMouseEvent* );

    /*! \brief When the user releases the button of the mouse
     */
    void contentsMouseReleaseEvent( QMouseEvent* );

    /*! \brief When the user double clicks on the button of the mouse.
     */
    void contentsMouseDoubleClickEvent( QMouseEvent* );
    
signals:
    
    /*! \brief Emit to change sheet when the user clicks on a state. 
     *  \param ID of the state.
     */
     void changeSheet( QString );
    
private:
     void contextMenuEvent ( QContextMenuEvent * );
     QDict<StateCanvas> dState;
     QCanvas *myCanvas;
     QCanvasItem *Moving;
     QPoint MovingStart;
     //QQueue<QPoint> MousePosQueue; 
     QString stateTerminalID;
     QString stateInitialID;
};

#endif  // __STATEMACHINEVIEW_H__
