/*
  name: statemachineview.cpp

  This file is part of ARCS - Augmented Reality Component System,
  written by Jean-Yves Didier, Vincent Le Ligeour and Yoann Petit
  for IBISC Laboratory (http://www.ibisc.univ-evry.fr)

  Copyright (C) 2004 Universit d'Evry-Val d'Essonne

  This program is free software: you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program.  If not, see <http://www.gnu.org/licenses/>.


  Please send bugreports  with examples or suggestions to
  didier__at__iup.univ-evry.fr
*/


#include "statemachineview.h"

#include <iostream>

#include "rtti.h"
#include <qwidget.h>
#include <qevent.h>
#include <qstrlist.h>
#include <qptrlist.h>
#include <qapplication.h>
#include <qwmatrix.h>
#include <qtoolbutton.h>
#include "iconsmac.h"

using namespace std;
/*
 Rtti_StateCanvas = 1003;
 Rtti_TransitionCanvas = 1004;
*/

StateMachineView::StateMachineView( QWidget *parent, const char *name , QString InitialID, QString TerminalID)
    : QCanvasView( parent, name )
{
     stateTerminalID = TerminalID;
     stateInitialID = InitialID;
     myCanvas = new QCanvas( this, "StateMachine" );
     myCanvas->setBackgroundColor( QColor(255,240,240));
     myCanvas->resize( 1500, 1500 );
     setCanvas( myCanvas );
     Moving = 0;
     dState.setAutoDelete(true);

     QToolButton* cornerImage = new QToolButton(this);
     cornerImage->setIconSet(QIconSet(QPixmap((const char**)iconsmac)));
     cornerImage->resize(20,20);
}


StateMachineView::~StateMachineView()
{
     Moving = NULL;
}


void StateMachineView::clear()
{
     stateTerminalID = QString();
     stateInitialID = QString();
     dState.clear();
}


void StateMachineView::contentsMousePressEvent( QMouseEvent *e )
{
     // a list of all items underneath the mouse when it was pressed
     QPoint posmap = (this->inverseWorldMatrix()).map(e->pos());

     QCanvasItemList list = canvas()->collisions(posmap);
     if ( !list.isEmpty() ) {
	  for( uint i = 0; i < list.count(); i++)
	  {
	       if (list[i]->rtti() == RTTI_STATECANVAS )
	       {
		    Moving = list[i];
		    MovingStart =  posmap;
		    break;
	       }
	  }
     }
}

void StateMachineView::contentsMouseDoubleClickEvent( QMouseEvent* e)
{
     QPoint posmap = (this->inverseWorldMatrix()).map(e->pos());

     QCanvasItemList list = canvas()->collisions(posmap);
     if ( !list.isEmpty() ) {
	  for( uint i = 0; i < list.count(); i++)
	  {
	       if (list[i]->rtti() == RTTI_STATECANVAS)
	       {
		    //cout<<"doubleclick"<<endl;
		    StateCanvas *sc = (StateCanvas*)list[i];
		    emit changeSheet(sc->getName());
		    break;
	       }
	  }
     }
}


void StateMachineView::contentsMouseMoveEvent( QMouseEvent *e )
{ 
     QPoint posmap = (this->inverseWorldMatrix()).map(e->pos());
     if ( Moving )    // si on selectionne un objet
     {	   
	  Moving->moveBy( posmap.x() - MovingStart.x(),
			  posmap.y() - MovingStart.y() );
	  MovingStart = posmap;
	  canvas()->update();   // mise a jour de notre canvas
     }
}

void StateMachineView::contentsMouseReleaseEvent( QMouseEvent* )
{
     if ( Moving )    // if we are holding the ball
	  Moving = 0;    
}

void StateMachineView::contextMenuEvent(QContextMenuEvent* e)
{
     QApplication::postEvent(parentWidget(), 
			     new QContextMenuEvent(e->reason(),e->pos(), e->globalPos(), e->state()));
}

/************************************************************************************/

void StateMachineView::registerState(StateCanvas *sc)
{
     dState.insert( sc->getName(), sc);
}

bool StateMachineView::isState(QString id)
{
     StateCanvas *result;     
     result = dState.find(id);
     if ( result != 0 ) 
	  return true;
     else
	  return false;
}

void StateMachineView::addTransition(TransitionCanvas *tc)
{
     dState[((StateCanvas*)(tc->getSender()))->getName()]->registerTransition(tc);
     dState[((StateCanvas*)(tc->getReceiver()))->getName()]->registerTransition(tc);
}

void StateMachineView::setInitialState(QString ID)
{
     if ( stateInitialID != ID ) {
	  if ( !stateInitialID.isEmpty() )
	       dState[stateInitialID]->setBrush( QColor(190,200,255));
	  stateInitialID = ID;
	  dState[stateInitialID]->setBrush( QColor(175,230,215) );
     }
     else {
	  dState[stateInitialID]->setBrush( QColor(190,200,255));
	  stateInitialID = QString();
     }
     if ( stateInitialID == stateTerminalID )
	  stateTerminalID = QString();
}

void StateMachineView::setTerminalState(QString ID)
{
     if ( stateTerminalID != ID ) {
	  if ( !stateTerminalID.isNull() )
	       dState[stateTerminalID]->setBrush( QColor(190,200,255));
	  stateTerminalID = ID;
	  dState[stateTerminalID]->setBrush( QColor(240,205,205) );
     }
     else {
	  dState[stateTerminalID]->setBrush( QColor(190,200,255));
	  stateTerminalID = QString();
     }
     if ( stateInitialID == stateTerminalID )
	  stateInitialID = QString();
}

void StateMachineView::renameStateID(QString ID, QString newID)
{
     StateCanvas *sc = dState[ID];
     if (sc == NULL)
	  return;
     sc->renameStateID(newID);
     dState.setAutoDelete(false);     
     dState.remove(ID);
     dState.setAutoDelete(true);
     dState.insert(newID, sc);
}


void StateMachineView::setTransition(MetaConfigWriter *exp)
{
     QDictIterator<StateCanvas> ite( dState );
     for( ; ite.current(); ++ite ) { 
	  ite.current()->setTransition(exp);
     }
}

void StateMachineView::delState(QString ID)
{
     StateCanvas* sc =  dState[ID];
     if (sc == NULL)
	  return;
     
    
     
     TransitionCanvas* tc1 = NULL;
     do
     {
	  QDictIterator<TransitionCanvas> ite = sc->getTransitionIterator();
	  //cout<<"parcours des transition sender : "<<ite.count()<<endl;
	  tc1 = ite.current();
	  if (ite.current())
	  {
	       StateCanvas* sc1 = (StateCanvas*)(ite.current()->getReceiver());
	       sc1->delTransitionReceiver(ite.current()); 
	       sc->delTransitionSender(ite.current());      
	  }
     } while (tc1);

     TransitionCanvas* tc2 = NULL;
     do
     {
	  QDictIterator<TransitionCanvas> ite2 = sc->getTransitionIteratorReceiver();
	  //cout<<"parcours des transition receiver : "<<ite2.count()<<endl;
	  tc2 = ite2.current();
	  if (ite2.current())
	  {
	       StateCanvas* sc2 = (StateCanvas*)(ite2.current()->getSender());
	       sc2->delTransitionSender(ite2.current());      
	       sc->delTransitionReceiver(ite2.current());
	  }
     } while (tc2);

     dState.remove(ID); 
} 
