/*
  name: mywidget.h

  This file is part of ARCS - Augmented Reality Component System,
  written by Jean-Yves Didier, Vincent Le Ligeour and Yoann Petit
  for IBISC Laboratory (http://www.ibisc.univ-evry.fr)

  Copyright (C) 2004 Universit d'Evry-Val d'Essonne

  This program is free software: you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program.  If not, see <http://www.gnu.org/licenses/>.


  Please send bugreports  with examples or suggestions to
  didier__at__iup.univ-evry.fr
*/


#ifndef __MYWIDGET_H__
#define __MYWIDGET_H__

/*! \file transitioncanvas.h
 * \author Yoann Petit
 * \date 7 Juillet 2005
 * 
 * File describes the main class
 */

#include <metalibrary/metabase.h>
#include <metalibrary/metalibrary.h>
#include <metalibrary/metastatemachine.h>

#include "sheetview.h"
#include "statemachineview.h"
#include "statecanvas.h"
#include "definesdialog.h"
#include "objectdialog.h"

#include "projectreader.h"

#include <qdict.h>
#include <qlabel.h>
#include <qmainwindow.h>
#include <qmenubar.h>
#include <qcanvas.h>

/*! \brief Main class description
 * 
 *  This is the main class
 */
class MyWidget : public QMainWindow
{
Q_OBJECT
public:

     /*! \brief ObjectCanvas constructor.
      *  \param *parent The parent of MyWidget.
      *  \param *name Name of this class.
      */
     MyWidget( QWidget *parent, const char *name );
     ~MyWidget();

     /*! \brief Sets the view, the StateMachine or a Sheet.
      *  \param numMode 1 for the StateMachine, 2 for a Sheet..
      */
     void setMode(int numMode);

     /*! \brief Resets some pointers
      */
     void resetPointers();

     /*! \brief List of fixed actions ID for the View menu.
      */
     enum MenuIDView{StateView, LastSheet};

     /*! \brief List of fixed actions ID for the State menu.
      */
     enum MenuIDState{AddSta, AddTra};

     /*! \brief List of fixed actions ID for the Sheet menu.
      */
     enum MenuIDSheet{AddSheet, DelSheet, RenSheet, AddLib, AddObj, AddIni, AddWir, OrderPre, OrderPost, OrderWire, AddMsl, AddMsi};

     enum MenuIDZoom{ ZoomIn, ZoomOut, Zoom11};


public slots:

     /*! \brief Clears the current project.
      */
     void clear();

     /*! \brief Exports the current project.
      */
     void Export();
     
     /*! \brief Opens a dialog to set the namefile for the export of the current project.
      */
     void ExportDial();

     /*! \brief Imports a project.
      * \param filename if a filename is given then it will import it.
      */
     void Import(QString fileName = QString::null);

     /*! \brief Opens a dialog to add an object on the current sheet.
      */
     void addObjectDial() { if (Sheet != NULL) ObjectDialog(0, "addObjectDialog", ml, Sheet).exec(); }

     /*! \brief Deletes a object of the current sheet.
      *  The object is also deleted in the application if it is not use on a other sheet.
      */
     void delObject();

     /*! \brief Opens a dialog to add a connection on the current sheet.
      */
     void addWireDial();

     /*! \brief Deletes a connection of the current sheet.
      *  The connection is also deleted in the application.
      */
     void delWire();

     /*! \brief Opens a dialog to add a pre or post-connection on the current sheet.
      */
     void addInitDial();

     /*! \brief Deletes a pre or post-connection of the current sheet.
      *  The pre or post-connection is also deleted in the application.
      */
     void delInit();


     void addMetaSignal();
     void addMetaSlot();

     void delIO();

     void switchObjectPersistence();


     void globalDefines() { defdiag->exec(); }


     void modifyInit();

     /*! \brief Opens a dialog to order post-connections on the current sheet.
      */
     void orderPostInitDial() { orderInitDial(true); }

     /*! \brief Opens a dialog to order pre-connections on the current sheet.
      */
     void orderPreInitDial() { orderInitDial(false); } 

     /*! \brief Opens a dialog to order connections on the current sheet.
      */
     void orderWireDial();

     /*! \brief Opens a dialog to add a state on the state machine.
      */
     void addStateDial();

     /*! \brief Opens a dialog to add a transition on the state machine.
      *  Call setNewState(QString s)
      */
     void addTransitionDial();

     /*! \brief Opens a dialog to add a new sheet in the application.
      */
     void addSheet();
     
     /*! \brief Deletes the current sheet.
      *  Call setNewSheet(QString s)
      */
     void delSheet();

     /*! \brief Deletes a state in the state machine but the sheet is not deleted in the application.
      */
     void delState() { if (!idStateSelect.isEmpty()) State->delState(idStateSelect); idStateSelect=QString::null; Canvas->update();}
     
     /*! \brief Changes the current sheet.
      *  \param ns Id of the new sheet.
      *  call changeSheet(QString s)
      */
     void changeSelect(QString ns);

     /*! \brief Changes the current sheet.
      *  \param sh Index of the new sheet in the View menu.
      *  call changeSheet(QString s)
      */
     void selectSheet(int sh);
     
     /*! \brief Renames the current sheet.
      */
     void renameSheet();

     /*! \brief Save the current view en picture format.
      */
     void saveCanvas();

     /*! \brief Opens a dialog to load a new library in the application.
      */
     void addLibrary();

     /*! \brief changes the name of the sheet.
      */
     void ChangeNameSheet(QString ns);
     
     /*! \brief Returns if the Id "is" is already use by a sheet.
      *  \param is ID of the new sheet.
      *  \param *s Boolean to return the anwser.
      */
     void hasSheet(QString is,bool *s) { if (dSheet.find(is)) *s = true; }


     /*! \brief Shows the state machine.
      */
     void StateMachine() { setMode(1); }

     /*! \brief Returns if the token "nt" for the state "ts" is already use by another transition.
      *  \param ts State source of the transition.
      *  \param nt Token of the transition.  
      *  \param *s Boolean to return the anwser.
      */
     void transitionIsPossible(QString ts, QString nt,bool *s) { if (((State->getStateCanvas(ts))->getTransition(nt))!= 0) *s=true; }


     /*! \brief Deletes a transition in the state machine.
      */
     void delTransition();
 
     /*! \brief Sets the selected object as the token sender of the current sheet.
      */
     void setTokenSender();

     /*! \brief Sets the selected state as the initial state of the state machine.
      */
     void setInitialState();

     /*! \brief Sets the selected state as the terminal state of the state machine.
      */
     void setTerminalState();
     
     /*! \brief Adds a new sheet.
      *  \param ns Id of the new sheet in the application.
      */
     void setNewSheet(QString ns); 

     /*! \brief Adds a new state.
      *  \param ns Id of the new state in the state machine.
      */
     void setNewState(QString ns) { State->registerState( new StateCanvas(Canvas,ns) );Canvas->update(); }


     /*! \brief Returns an iterator to the dictionnary of SheetView.
      *  \return Iterator to the dictionnary of SheetView.
      */
     QDictIterator<SheetView> getSheetIterator() { return QDictIterator<SheetView>(dSheet);}


     void refresh(); //!< \brief refresh the canvas view.
     
     void run();//!< \brief Starts the program metaruntime with application currently being designed

     /*! \brief Apply a relative zoom factor on canvas.
      * \param zf if \a zf is -1 then zoom int twice, if \a zf is 1 then zoom out twice, if \a zf is 0 then go back to scale 1:1
      */
     void zoom(int zf);

     void about();
     
private:
     void createMetablockSheet(MetaBlock* mb0);
     ObjectCanvas* checkObject(QString objname, int& idx);
     void autoPlaceObject(ObjectCanvas* oc, int idx);
     void autoPlaceState(StateCanvas* sc, int idx);
     void importIO();
     void importElements();
     void importStateMachine();
     void importGraphicsSheet(ProjectReader* prd);
     void importGraphicsBlock(QString name);

     void exportBlocks();
     void importSheet(MetaSheet* ms);
     bool importApplication(QString file);
     MetaSheet* importBlock(QString file);
     void changeSheet(QString s);
     QString getType(QString s);
     void contextMenuEvent ( QContextMenuEvent * );
     void orderInitDial(bool typeOfInit);
     QRect computeCanvasBoundingBox(QCanvas* canvas);

     MetaLibrary ml;
     MetaStateMachine msm;
     ObjectCanvas *ObjectSelect;
     StateCanvas *StateSelect;
     InitCanvas *InitSelect;
     WireCanvas *WireSelect;
     IOCanvas* IOSelect;
     TransitionCanvas *TransitionSelect;
     QString saveName; //< Nom du fihcier dans lequel la sauvegarde est effectue.
     DefinesDialog* defdiag;
     
     QDict<SheetView> dSheet; //!< dictionnaire des feuilles
     QMenuBar *Menu; //!< Menu de l'application 
     QCanvas *Canvas; //!< Canvas principal de l'application
     SheetView *Sheet; //!< Feuille courante
     StateMachineView *State; //!< Automate 
     QPopupMenu *view; //!< Menu 
     QPopupMenu *menuZoom; //!< Zooming options
     QPopupMenu *menuSheet; //!< Menu Sheet
     QPopupMenu *menuState; //!< Menu State
     QPopupMenu *contextMenu; //!< Menu apparaissant lors d'un click droite de la souris
     QLabel *viewActive; //!< Nom de la vue active(L'automate ou une des feuilles)
     QString idObjectToken; //!< 
     QString idStateSelect; //!<
     int indexSheet; //!<
     int numIndex; //!<
}; 

#endif  // __MYWIDGET_H__
