/*
  name: definesdialog.cpp

  This file is part of ARCS - Augmented Reality Component System,
  written by Jean-Yves Didier, Vincent Le Ligeour and Yoann Petit
  for IBISC Laboratory (http://www.ibisc.univ-evry.fr)

  Copyright (C) 2004 Universit d'Evry-Val d'Essonne

  This program is free software: you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program.  If not, see <http://www.gnu.org/licenses/>.


  Please send bugreports  with examples or suggestions to
  didier__at__iup.univ-evry.fr
*/


#include "definesdialog.h"

#include <qvbox.h>
#include <qmessagebox.h>

DefinesDialog::DefinesDialog(QWidget* parent, const char* name)
      : QDialog( parent, name, TRUE, 0)
{
     setCaption("Add a new variable");
     colHeaders << "Name" << "Type" << "Value" ;

     QVBox *vb1 = new QVBox( this );
     vb1->resize( 350, 300 );
     table = new QTable( vb1 );
     table->setNumCols(3);
     table->setSelectionMode(QTable::SingleRow);
     table->setColumnLabels(colHeaders);

     QHBox *hb0 = new QHBox( vb1 );

     newDefine = new QPushButton("New...",hb0);
     newDefine->setMaximumWidth( 80 );
     connect( newDefine, SIGNAL(clicked()), SLOT(addNewLine()) );

     delDefine = new QPushButton("Delete",hb0);
     newDefine->setMaximumWidth( 80 );
     connect( delDefine, SIGNAL(clicked()), SLOT(delLine()) );

     

     ok = new QPushButton( "Ok", hb0 );
     ok->setMaximumWidth( 80 );
     connect( ok, SIGNAL(clicked()), SLOT(commitChanges()) );

     cancel = new QPushButton( "Cancel", hb0 );
     cancel->setMaximumWidth(80);
     connect( cancel, SIGNAL(clicked()), SLOT(reject()) );

     connect(table, SIGNAL(valueChanged(int,int)), this, SLOT(changeValue(int,int)));

}


void DefinesDialog::addNewLine()
{
     QStringList typeList;

     typeList << "string" << "int" << "float" << "double" << "bool" << "path"; 

     table->insertRows(table->numRows());

     table->setItem(table->numRows()-1, 1, new QComboTableItem(table, typeList));
}


void DefinesDialog::delLine()
{
     if (table->numSelections() == 0)
	  return;

     
     if ( QMessageBox::Yes == QMessageBox::warning(this, "Delete variable", 
						   "Are you sure to delete the selected variable ?",
						   QMessageBox::Yes,QMessageBox::No))
     {
	  table->removeRow(table->currentRow());
     }
}


void DefinesDialog::show()
{
     table->setNumRows(0);
     fill();
     QDialog::show();
}


void DefinesDialog::changeValue(int row, int col)
{
     if (col != 1)
	  return ;

     QString type = ((QComboTableItem*)(table->item(row,col)))->currentText();


     if (type == "bool" )
     {
	  table->setItem(row,2, new QCheckTableItem(table, "true"));
	  return;
     }


     if (type == "int")
     {
	  table->setItem(row,2, new QIntTableItem(table,QTableItem::WhenCurrent));
	  return;
     }

     if ((type == "float") || (type == "double"))
     {
	  table->setItem(row,2, new QFloatTableItem(table,QTableItem::WhenCurrent));
	  return;
     }
    
     table->setItem(row,2, new QTableItem(table,QTableItem::Always));
}




void DefinesDialog::commitChanges()
{
     MetaDefine::clear();

     std::cout << "Commiting " << table->numRows() << " changes" << std::endl;
     for (int j=0; j < table->numRows(); j++)
     {
	  if (table->text(j,0).isEmpty())
	  {
	       QMessageBox::information(this,"empty variable name",
					"At least one variable name is empty.\n You should name it first.",
					QMessageBox::Ok);
	       return;
	  }

     }


     for (int i =0; i < table->numRows(); i++)
	  new MetaDefine(table->text(i,0), table->text(i,1), table->text(i,2));

     accept();
}


void DefinesDialog::fill()
{
     QDictIterator<MetaDefine> ite = MetaDefine::getIterator();

     for( ; ite.current(); ++ite )
     {
	  addNewLine();
	  table->setText(table->numRows()-1,0, ite.currentKey());
	  QComboTableItem* cti = (QComboTableItem*)table->item(table->numRows()-1,1);
	  cti->setCurrentItem(ite.current()->getType());
	  table->setText(table->numRows()-1,2, ite.current()->getValue());
     }
}
